<?php declare(strict_types = 1);

namespace Ceskaposta\Model;

use Ceskaposta\Model\Entities\ICeskaPostaOrder;
use Ceskaposta\Model\Entities\PostOfficeOrder;
use Ceskaposta\Model\Entities\PostWarehouseOrder;
use Core\Model\Helpers\Strings;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\Order;
use Nette\Utils\Validators;

class AdvancedOptions
{
	public const VARIABLE_SYMBOL_AS_INVOICE_NUMBER = 'invoiceNumber';
	public const VARIABLE_SYMBOL_AS_ORDER_NUMBER   = 'orderNumber';
	protected Order $order;

	public function __construct(
		protected ICeskaPostaOrder $cpOrder,
		protected array            $options,
	)
	{
		$this->order = $this->cpOrder->getOrder();
	}

	public function getWeight(): string
	{
		if (isset($this->options['weight']) && Validators::isNumericInt($this->options['weight'])) {
			return str_replace(',', '.', (string) $this->options['weight']);
		}

		if ($this->order->getItemsWeight() !== 0.0) {
			return str_replace(',', '.', (string) $this->order->getItemsWeight());
		}

		return str_replace(',', '.', (string) CeskaPostaConfig::load('weight'));
	}

	public function getInsuredValue(): float
	{
		$price = $this->order->getPriceWithoutDiscounts();
		if (isset($this->options['insuredValue']) && Validators::isNumber($this->options['insuredValue'])) {
			$price = (float) Strings::formatEntityDecimal($this->options['insuredValue']);
		}

		if (strtolower($this->order->getCurrencyCode()) === 'czk') {
			$price = ceil($price);
		}

		return $price;
	}

	public function getExportNote(): ?string
	{
		if (isset($this->options['exportNote']) && strlen((string) $this->options['exportNote']) > 0) {
			return $this->options['exportNote'];
		}

		return null;
	}

	public function getVariableSymbol(): string
	{
		if (isset($this->options['variableSymbol']) && $this->options['variableSymbol'] === self::VARIABLE_SYMBOL_AS_INVOICE_NUMBER && $this->order->getInvoice() instanceof Invoice) {
			return $this->order->getInvoice()->ident;
		}

		return (string) $this->order->getId();
	}

	public function getServices(int $quantityParcel): array
	{
		$this->options['services'] = isset($this->options['services']) && is_array($this->options['services']) ? $this->options['services'] : [];
		$this->options['services'] = array_map('intval', $this->options['services']);
		$isCod                     = $this->order->getPaymentIdent() === 'cod';
		$defaultServices           = CeskaPostaConfig::load('services', []);
		$services                  = array_merge(array_map(static fn($val): mixed => is_numeric($val) ? ((int) $val) : $val, $defaultServices), [7]);

		if ($isCod) {
			$services[] = 41;
		}
		if ($quantityParcel > 1) {
			$services[] = 70;
		}
		$services           = array_merge($services, $this->options['services']);
		$defaultPackageSize = ['M'];
		$services           = array_merge($services, isset($this->options['packageSize']) && !empty($this->options['packageSize']) && (string) $this->options['packageSize'] !== '' ? [strtoupper((string) $this->options['packageSize'])] : $defaultPackageSize);

		return array_unique($services);
	}

	public function getUnifiedPrefix(): string
	{
		$prefix = Strings::upper($this->cpOrder->getPrefixParcelCode());

		if ($prefix === PostOfficeOrder::PREFIX_PARCEL_CODE) {
			return PostWarehouseOrder::PREFIX_PARCEL_CODE;
		}

		return $prefix;
	}

}
