<?php declare(strict_types = 1);

namespace Ceskaposta\Model\Entities;

use Ceskaposta\Model\CeskaPostaConfig;
use Ceskaposta\Model\Entities\Attrs\IdTransaction;
use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Listeners\CarrierListener;

#[ORM\Table(name: 'ceskaposta__parcel_delivery_to_hand_order')]
#[ORM\Entity]
#[ORM\EntityListeners([CarrierListener::class])]
class ParcelDeliveryToHandOrder implements ICeskaPostaOrder
{
	public const PREFIX_PARCEL_CODE = 'DR';

	use IdTransaction;

	#[ORM\Id]
	#[ORM\OneToOne(targetEntity: Order::class)]
	#[ORM\JoinColumn(name: 'order_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	protected Order $order;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	private ?DateTime $exported = null;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $numberPackage = null;

	/** @var Collection<string, ParcelNumberDeliveryToHand> */
	#[ORM\OneToMany(mappedBy: 'parcelOrder', targetEntity: ParcelNumberDeliveryToHand::class, cascade: ['all'], orphanRemoval: true, indexBy: 'numberPackage')]
	public Collection $associatedNumberPackages;

	public function __construct(Order $order)
	{
		$this->order                    = $order;
		$this->associatedNumberPackages = new ArrayCollection;
	}

	public function getOrder(): Order { return $this->order; }

	public function export(?DateTime $datetime = null): self
	{
		$this->exported = $datetime ?? new DateTime;

		return $this;
	}

	public function getExported(): ?DateTime { return $this->exported; }

	public function getPostName(): ?string
	{
		return null;
	}

	public function getVillage(): ?string
	{
		return null;
	}

	public function getPartVillage(): ?string
	{
		return null;
	}

	public function getNumberPackage(): ?string
	{
		return $this->numberPackage;
	}

	public function isExported(): bool
	{
		return $this->getExported() !== null;
	}

	public function isPickUp(): bool
	{
		return true;
	}

	public function getPostPSC(): ?string
	{
		return null;
	}

	public function getPrefixParcelCode(): string
	{
		return static::PREFIX_PARCEL_CODE;
	}

	public function getTrackingUrl(): string
	{
		return $this->numberPackage ? sprintf(CeskaPostaConfig::load('trackingUrl', ''), $this->numberPackage) : '';
	}

	public function getTrackingUrls(): array
	{
		$result = [];
		if (empty(!$this->getTrackingUrl())) {
			$result[$this->numberPackage] = $this->getTrackingUrl();
		}
		foreach ($this->getAssociatedNumberPackages() as $number) {
			$result[$number->numberPackage] = sprintf(CeskaPostaConfig::load('trackingUrl', ''), $number->numberPackage);
		}

		return $result;
	}

	public function resetExport(): void
	{
		$this->numberPackage = null;
		$this->exported      = null;
		$this->idTransaction = null;
		$this->associatedNumberPackages->clear();
	}

	public function getIdent(): string
	{
		return 'doruky';
	}

	/**
	 * @return ParcelNumberDeliveryToHand[]
	 */
	public function getAssociatedNumberPackages(): array
	{
		return $this->associatedNumberPackages->toArray();
	}

	public function getAllNumberPackages(): array
	{
		if (!$this->numberPackage) {
			return [];
		}

		return array_merge([$this->numberPackage], array_map(static fn(ICeskaPostaNumberPackage $parcel) => $parcel->getNumberPackage(), $this->getAssociatedNumberPackages()));
	}

}
