<?php declare(strict_types = 1);

namespace Ceskaposta\Model\Entities;

use Ceskaposta\Model\CeskaPostaConfig;
use Ceskaposta\Model\Entities\Attrs\IdTransaction;
use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Listeners\CarrierListener;

#[ORM\Table(name: 'ceskaposta__post_office_order')]
#[ORM\Entity]
#[ORM\EntityListeners([CarrierListener::class])]
class PostOfficeOrder implements ICeskaPostaOrder
{
	public const PREFIX_PARCEL_CODE = 'NP';

	use IdTransaction;

	#[ORM\JoinColumn(name: 'order_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\Id]
	#[ORM\OneToOne(targetEntity: Order::class)]
	protected Order $order;

	#[ORM\Column(name: 'post_psc', type: Types::STRING)]
	public string $postPSC;

	#[ORM\Column(type: Types::STRING)]
	public string $postName;

	#[ORM\Column(type: Types::STRING)]
	public string $village;

	#[ORM\Column(type: Types::STRING)]
	public string $partVillage;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	private ?DateTime $exported = null;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $numberPackage = null;

	/** @var Collection<string, ParcelNumberPostOffice> */
	#[ORM\OneToMany(mappedBy: 'parcelOrder', targetEntity: ParcelNumberPostOffice::class, cascade: ['all'], orphanRemoval: true, indexBy: 'numberPackage')]
	public Collection $associatedNumberPackages;

	public function __construct(Order $order, string $postPSC)
	{
		$this->order                    = $order;
		$this->postPSC                  = $postPSC;
		$this->associatedNumberPackages = new ArrayCollection;
	}

	public function getOrder(): Order { return $this->order; }

	public function getPostPSC(): string { return $this->postPSC; }

	public function export(?DateTime $datetime = null): self
	{
		$this->exported = $datetime ?? new DateTime;

		return $this;
	}

	public function getExported(): ?DateTime
	{
		return $this->exported;
	}

	public function getPostName(): ?string
	{
		return $this->postName;
	}

	public function getVillage(): ?string
	{
		return $this->village;
	}

	public function getPartVillage(): ?string
	{
		return $this->partVillage;
	}

	public function getNumberPackage(): ?string
	{
		return $this->numberPackage;
	}

	public function isExported(): bool
	{
		return $this->getExported() !== null;
	}

	public function isPickUp(): bool
	{
		return true;
	}

	public function getPrefixParcelCode(): string
	{
		return static::PREFIX_PARCEL_CODE;
	}

	public function getTrackingUrl(): string
	{
		return $this->numberPackage ? sprintf(CeskaPostaConfig::load('trackingUrl', ''), $this->numberPackage) : '';
	}

	public function getTrackingUrls(): array
	{
		$result = [];
		if (empty(!$this->getTrackingUrl())) {
			$result[$this->numberPackage] = $this->getTrackingUrl();
		}
		foreach ($this->getAssociatedNumberPackages() as $number) {
			$result[$number->numberPackage] = sprintf(CeskaPostaConfig::load('trackingUrl', ''), $number->numberPackage);
		}

		return $result;
	}

	public function resetExport(): void
	{
		$this->numberPackage = null;
		$this->exported      = null;
		$this->idTransaction = null;
		$this->associatedNumberPackages->clear();
	}

	public function getIdent(): string
	{
		return 'napostu';
	}

	/**
	 * @return ParcelNumberPostOffice[]
	 */
	public function getAssociatedNumberPackages(): array
	{
		return $this->associatedNumberPackages->toArray();
	}

	public function getAllNumberPackages(): array
	{
		if (!$this->numberPackage) {
			return [];
		}

		return array_merge([$this->numberPackage], array_map(static fn(ICeskaPostaNumberPackage $parcel) => $parcel->getNumberPackage(), $this->getAssociatedNumberPackages()));
	}
}
