<?php declare(strict_types = 1);

namespace Ceskaposta\Model\Libs;

use Ceskaposta\Model\CeskaPostaConfig;
use Ceskaposta\Model\Utils\Validators;
use EshopOrders\Model\ExpeditionLogger;
use Exception;
use Nette\Schema\Expect;
use Nette\Schema\Processor;
use Nette\Utils\Arrays;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Nette\Utils\Json;
use SimpleXMLElement;

class CeskaPostaApiBridge
{
	protected CPOSTOLD $cpostOld;
	protected CPOST    $cpost;
	protected array    $config;

	public function __construct(
		protected ExpeditionLogger $expeditionLogger,
	)
	{
		$config = [
			'enable'             => CeskaPostaConfig::load('enable'),
			'contractId'         => CeskaPostaConfig::load('contractId'),
			'customerId'         => CeskaPostaConfig::load('customerId'),
			'postCode'           => CeskaPostaConfig::load('postCode'),
			'locationNumber'     => CeskaPostaConfig::load('locationNumber'),
			'certFile'           => CeskaPostaConfig::load('certFile'),
			'certPassword'       => CeskaPostaConfig::load('certPassword'),
			'label'              => CeskaPostaConfig::load('label'),
			'services'           => CeskaPostaConfig::load('services'),
			'weight'             => CeskaPostaConfig::load('weight'),
			'allowGenerateLabel' => CeskaPostaConfig::load('allowGenerateLabel'),
			'trackingUrl'        => CeskaPostaConfig::load('trackingUrl'),
			'transmissionDate'   => CeskaPostaConfig::load('transmissionDate'),
			'useCa5'             => CeskaPostaConfig::load('useCa5'),
			'debug'              => CeskaPostaConfig::load('debug'),
			'enableNewApi'       => CeskaPostaConfig::load('enableNewApi'),
			'apiToken'           => CeskaPostaConfig::load('apiToken'),
			'secretKey'          => CeskaPostaConfig::load('secretKey'),
		];

		if (CeskaPostaConfig::load('enable')) {
			$cert = CeskaPostaConfig::load('enableNewApi') ? '' : static::getCertFile($config['certFile']);
			static::validateConfig($config);
			$this->config = $config;

			if (!CeskaPostaConfig::load('enableNewApi')) {
				$this->cpostOld        = new CPOSTOLD($config['contractId'], $cert, $config['certPassword'], (bool) $config['useCa5']);
				$this->cpostOld->debug = $config['debug'];
			}

			$this->cpost = new CPOST($config['contractId'], $config['apiToken'], $config['secretKey'], (bool) $config['debug']);
		}
	}

	protected static function getCertFile(string $originalCertFile): string
	{
		$targetCertFile = TMP_DIR . DS . 'cert' . DS . basename($originalCertFile);
		if (!file_exists($targetCertFile)) {
			FileSystem::copy($originalCertFile, $targetCertFile);
		}

		return $targetCertFile;
	}

	protected static function validateConfig(array $config): void
	{
		$keyFnDesc  = 'If you use new api, api token and secret key must be filled.';
		$keyFn      = static fn($val): bool => $config['enableNewApi'] === false || (!Validators::isNone($config['apiToken']) && !Validators::isNone($config['secretKey']));
		$certFnDesc = 'If you use old api, cert file and cert password must be filled.';
		$certFn     = static fn($val): bool => $config['enableNewApi'] === true || (!Validators::isNone($config['certFile']) && is_file($config['certFile']) && !Validators::isNone($config['certPassword']));

		$schema = Expect::structure([
			'enable'             => Expect::anyOf(true),
			'enableNewApi'       => Expect::bool(),
			'apiToken'           => Expect::string()->assert($keyFn, $keyFnDesc),
			'secretKey'          => Expect::string()->assert($keyFn, $keyFnDesc),
			'contractId'         => Expect::string()->min(1),
			'customerId'         => Expect::string(),
			'postCode'           => Expect::string(),
			'locationNumber'     => Expect::int(),
			'certFile'           => Expect::string()->assert($certFn, $certFnDesc),
			'certPassword'       => Expect::string()->assert($certFn, $certFnDesc),
			'label'              => Expect::structure([
				'idForm'          => Expect::int()->nullable(),
				'shiftHorizontal' => Expect::int(),
				'shiftVertical'   => Expect::int(),
				'position'        => Expect::int()->nullable(),
			]),
			'services'           => Expect::listOf('int|string'),
			'weight'             => Expect::type('int|float'),
			'allowGenerateLabel' => Expect::bool(),
			'trackingUrl'        => Expect::string(),
			'transmissionDate'   => Expect::string()->nullable()->assert(static function($val) {
				if ($val instanceof DateTime) {
					return true;
				}

				try {
					new DateTime($val);

					return true;
				} catch (Exception) {
					return false;
				}
			}),
			'useCa5'             => Expect::bool(),
			'debug'              => Expect::bool(),
		]);

		(new Processor)->process($schema, $config);
	}

	/**
	 * Async call. Returns idTransaction
	 *
	 * @throws CeskaPostaException
	 */
	public function sendParcels(array $data): ?string
	{
		if ($this->config['enableNewApi']) {
			$parcelHeader = $data['doParcelHeader'];
			if (isset($parcelHeader['transmissionDate']) && Validators::isDateTime($parcelHeader['transmissionDate'])) {
				$parcelHeader['transmissionDate'] = (new DateTime($parcelHeader['transmissionDate']))->format('Y-m-d');
			}

			$parcelDataList = $data['doParcelData'];
			foreach ($parcelDataList as $k => $v) {
				Arrays::renameKey($parcelDataList[$k], 'doParcelParams', 'parcelParams');
				Arrays::renameKey($parcelDataList[$k], 'doParcelAddress', 'parcelAddress');
				Arrays::renameKey($parcelDataList[$k], 'doParcelServices', 'parcelServices');

				$this->normalizeServices($parcelDataList[$k]['parcelServices']);
				$this->normalizeParcelAddress($parcelDataList[$k]['parcelAddress']);
				$this->normalizeParcelParams($parcelDataList[$k]['parcelParams']);
			}

			$response = $this->cpost->sendParcels($parcelHeader, $parcelDataList);

			if ($response['idTransaction']) {
				return $response['idTransaction'];
			}

			return null;
		}

		$response = $this->cpostOld->sendParcels($data);

		if (isset($response->header->idTransaction)) {
			return (string) $response->header->idTransaction;
		}

		return null;
	}

	public function sendParcelsSync(array $data): array
	{
		if ($this->config['enableNewApi']) {

			if (isset($data['doPOLSyncParcelHeader']['transmissionDate']) && Validators::isDateTime($data['doPOLSyncParcelHeader']['transmissionDate'])) {
				$data['doPOLSyncParcelHeader']['transmissionDate'] = (new DateTime($data['doPOLSyncParcelHeader']['transmissionDate']))->format('Y-m-d');
			}

			$parcelServiceHeader = ['parcelServiceHeaderCom' => $data['doPOLSyncParcelHeader']];
			$parcelServiceData   = $data['doPOLSyncParcelData'];

			Arrays::renameKey($parcelServiceData, 'doPOLSyncParcelParams', 'parcelParams');
			Arrays::renameKey($parcelServiceData, 'doPOLParcelAddress', 'parcelAddress');
			$parcelServiceData['parcelServices'] = $parcelServiceData['parcelParams']['doPOLParcelServices'];
			unset($parcelServiceData['parcelParams']['doPOLParcelServices']);

			$this->normalizeServices($parcelServiceData['parcelServices']);
			$this->normalizeParcelAddress($parcelServiceData['parcelAddress']);
			$this->normalizeParcelParams($parcelServiceData['parcelParams']);

			$response       = $this->cpost->parcelService($parcelServiceHeader, $parcelServiceData);
			$responseHeader = $response['responseHeader']['resultHeader'];
			$responseCode   = (int) $responseHeader['responseCode'];

			if ($responseCode !== 1) {
				$this->logParcelDataError($response['responseHeader']['resultParcelData']);
			}

			return $response;
		}

		$response = Json::decode(Json::encode((array) $this->cpostOld->sendParcelsSync($data)), Json::FORCE_ARRAY);

		if (!isset($response['responseHeader']['resultParcelData'][0]) && isset($response['responseHeader']['resultParcelData']['recordNumber'], $response['responseHeader']['resultParcelData']['parcelCode'])) {
			$recordNumber = $response['responseHeader']['resultParcelData']['recordNumber'];
			$parcelCode   = $response['responseHeader']['resultParcelData']['parcelCode'];
			foreach ($response['responseHeader']['resultParcelData'] as $k => $v) {
				unset($response['responseHeader']['resultParcelData'][$k]);
			}
			$response['responseHeader']['resultParcelData'][0]['recordNumber'] = $recordNumber;
			$response['responseHeader']['resultParcelData'][0]['parcelCode']   = $parcelCode;
		}

		return $response;
	}

	public function getResultParcels(string $idTransaction): array
	{
		if ($this->config['enableNewApi']) {
			$response = $this->cpost->sendParcelsGet($idTransaction);

			$responseCode = (string) $response['StatusResponseList'][0]['responseCode'];

			if ($responseCode === '19') {
				$this->logParcelDataError($response['ResultSendParcelsList']);
			}

			return [
				'errorCode' => $responseCode,
				'parcels'   => $response['ResultSendParcelsList'],
			];
		}

		/** @var SimpleXMLElement $response */
		$response    = $this->cpostOld->getResultParcels($idTransaction);
		$responseArr = Json::decode(Json::encode((array) $response), Json::FORCE_ARRAY);

		return [
			'errorCode' => (string) (empty($response->doParcelParamResult) ? $responseArr['errorCode'] : $response->doParcelHeaderResult->doParcelStateResponse->responseCode),
			'parcels'   => empty($response->doParcelParamResult) ? [] : $responseArr['doParcelParamResult'],
		];
	}

	public function getParcelState(string $idParcel): array
	{
		if ($this->config['enableNewApi']) {
			$response = $this->cpost->parcelStatus([$idParcel]);

			return $response['detail'][0]['parcelStatuses'] ?? [];
		}

		$response    = $this->cpostOld->getParcelState($idParcel);
		$responseArr = Json::decode(Json::encode((array) $response), Json::FORCE_ARRAY);

		return empty($response->states) ? [] : $responseArr['states']['state'];
	}

	public function getParcelsPrinting(array $data): array
	{
		if ($this->config['enableNewApi']) {
			$customerId      = $data['doPrintingHeader']['customerID'];
			$codes           = $data['doPrintingData']['parcelCode'];
			$idForm          = $data['doPrintingHeader']['idForm'];
			$shiftHorizontal = $data['doPrintingHeader']['shiftHorizontal'];
			$shiftVertical   = $data['doPrintingHeader']['shiftVertical'];
			$position        = $data['doPrintingHeader']['position'];

			$response = $this->cpost->parcelPrinting($customerId, $codes, $idForm, $position, $shiftHorizontal, $shiftVertical);
//			if (isset($response['printingDataResult'])) {
//				$response['printingDataResult']['file'] = $response['printingDataResult'];
//			}

			return [
				'data' => $response['printingHeaderResult'],
				'file' => $response['printingDataResult'],
			];
		}

		$response    = $this->cpostOld->getParcelsPrinting($data);
		$responseArr = Json::decode(Json::encode((array) $response), Json::FORCE_ARRAY);

		Arrays::renameKey($responseArr['doPrintingHeaderResult'], 'doPrintingHeader', 'printingHeader');
		Arrays::renameKey($responseArr['doPrintingHeaderResult'], 'doPrintingStateResponse', 'printingStateResponse');
		Arrays::renameKey($responseArr, 'doPrintingHeaderResult', 'printingHeaderResult');
		Arrays::renameKey($responseArr, 'doPrintingDataResult', 'printingDataResult');

		return [
			'data' => $responseArr['printingHeaderResult'],
			'file' => $responseArr['printingDataResult']['file'],
		];
	}

	/**
	 * @return DateTime|string
	 */
	public function getTransmissionDate()
	{
		return $this->config['transmissionDate'];
	}

	public function getConfig(): array
	{
		return $this->config;
	}

	protected function normalizeServices(array &$array): void
	{
		foreach ($array as $key => $service) {
			$s = $service['service'];
			unset($service['service']);
			$array[$key] = (string) $s;
		}
	}

	protected function normalizeParcelAddress(array &$array): void
	{
		foreach ($array as $key => $addressVal) {
			switch ($key) {
				case 'isoCountry':
				case 'city':
				case 'zipCode':
				case 'houseNumber':
				case 'street':
					unset($array[$key]);
					$array['address'][$key] = $addressVal;
					break;
				case 'companyName':
					unset($array[$key]);
					$array['company'] = $addressVal;
					break;
				case 'mobileNumber':
					unset($array[$key]);
					$array['mobilNumber'] = $addressVal;
					break;
				case 'recordID':
					$array[$key] = (string) $addressVal;
					break;
			}
		}
	}

	protected function normalizeParcelParams(array &$array): void
	{
		if (isset($array['recordID'])) {
			$array['recordID'] = (string) $array['recordID'];
		}

		if (isset($array['parcelCode']) && Validators::isNone($array['parcelCode'])) {
			unset($array['parcelCode']);
		}
	}

	protected function logParcelDataError(array $result): void
	{
		$errors = [];

		foreach ($result as $d) {
			$orderId = (int) $d['recordNumber'];

			if (!array_key_exists('parcelStateResponse', $d) || !is_array($d['parcelStateResponse'])) {
				continue;
			}

			foreach ($d['parcelStateResponse'] as $e) {
				$err = sprintf('%s - %s', $e['responseCode'], $e['responseText']);

				if (!in_array($err . ' - ' . $orderId, $errors)) {
					$errors[] = $err . ' - ' . $orderId;
					$this->expeditionLogger->logError('cpost', $err, $orderId);
				}
			}
		}
	}

}
