<?php declare(strict_types = 1);

namespace Ceskaposta\AdminModule\Model\Subscribers;

use Ceskaposta\Model\CeskaPostaConfig;
use Ceskaposta\Model\Entities\PostWarehouseOrder;
use Ceskaposta\Model\Libs\CeskaPostaApi;
use Ceskaposta\Model\OrdersExported;
use Contributte\Translation\Translator;
use Core\Model\Event\ControlEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Helpers\Arrays;
use EshopOrders\AdminModule\Components\Expedition\ExpeditionAdvancedOptionsForm;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\AdminModule\Model\Event\OrderNavigationCallbackEvent;
use EshopOrders\AdminModule\Model\Event\OrdersNavigationEvent;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\Model\EshopOrdersConfig;
use Override;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class CeskapostaOrdersSubscriber implements EventSubscriberInterface
{
	public function __construct(
		protected OrdersExported $ordersExported,
		protected CeskaPostaApi  $apiService,
		protected Translator     $translator,
		protected Speditions     $speditionsService,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.admin.ordersDefaultNavigation'                  => 'ordersDefaultNavigation',
			'eshopOrders.admin.ordersDefaultNavigationCallback'          => 'ordersDefaultNavigationCallback',
			OrdersGrid::class . '::onAttach'                             => ['ordersGridAttached', 255],
			ExpeditionAdvancedOptionsForm::class . '::formSuccess-cpost' => ['expeditionAdvancedOptionsFormSuccess',
				255],
		];
	}

	public function ordersDefaultNavigation(OrdersNavigationEvent $event): void
	{
		if (EshopOrdersConfig::load('hideOrderGridNavExport') || !CeskaPostaConfig::load('enable')) {
			return;
		}

		$event->addNavigation(
			'ceskaposta.ordersNavigation.exportAll',
			['navigationCallback!', ['key' => 'ceskapostaExport']],
			'fas fa-truck-loading',
			'ajax'
		);
	}

	public function ordersGridAttached(ControlEvent $event): void
	{
		/** @var OrdersGrid $control */
		$control = $event->control;

		if (CeskaPostaConfig::load('enable')) {

			if (EshopOrdersConfig::load('expeditionOrdersGrid.enableAdvancedOptions', false)) {
				$control->addExportCallback('ceskaposta', 'ceskaposta.ordersGrid.export', function(array $ids) use ($control): void {
					$control->openExpeditionAdvancedOptionsFormModal($ids, 'cpost');
				});
			} else {
				$control->addExportCallback('ceskaposta', 'ceskaposta.ordersGrid.export', function(array $ids, $quantity) use ($control): void {
					$orders = $this->ordersExported->getOrdersNotExported($ids);
					if ($orders === []) {
						$control->presenter->flashMessageSuccess($this->translator->translate('ceskaposta.export.noOrders'));
						$control->presenter->redrawControl('flashes');

						return;
					}

					if (((int) $quantity) > (int) CeskaPostaConfig::load('maxPackages')) {
						$control->presenter->flashMessageWarning($this->translator->translate('ceskaposta.export.maxPackages', ['quantity' => CeskaPostaConfig::load('maxPackages')]));
						$control->presenter->redrawControl('flashes');

						return;
					}

					if (((int) $quantity) > 1 && Arrays::some($orders, static fn($order): bool => $order instanceof PostWarehouseOrder)) {
						$control->presenter->flashMessageWarning($this->translator->translate('ceskaposta.export.multiplePiecesOrderWarehouse'));
						$control->presenter->redrawControl('flashes');

						return;
					}

					set_time_limit(120);
					$result = $this->apiService->sendOrders($orders, max(1, (int) $quantity));

					if (isset($result['message']) && is_array($result['message'])) {
						foreach ($result['message'] as $messages) {
							if (!is_array($messages)) {
								continue;
							}
							foreach ($messages as $k => $m) {
								match ($k) {
									'danger' => $control->presenter->flashMessageDanger($m),
									'warning' => $control->presenter->flashMessageWarning($m),
									default => $control->presenter->flashMessageInfo($m),
								};
							}
						}
					}

					if ($result['error'] === 0) {
						$control->presenter
							->flashMessageSuccess($this->translator->translate('ceskaposta.export.success'));
					} else if ($result['error'] > 0 && $result['ok'] > 0) {
						$control->presenter
							->flashMessageWarning($this->translator->translate('ceskaposta.export.partial', null, $result));
					} else {
						$control->presenter
							->flashMessageDanger($this->translator->translate('ceskaposta.export.error'));
					}

					$control->presenter->redrawControl('flashes');
				});
			}

			if (CeskaPostaConfig::load('allowGenerateLabel', false)) {
				$control->addExportCallback('ceskapostaLabels', 'ceskaposta.ordersGrid.labels', function(array $ids) use ($control): void {

					$orders = $this->ordersExported->getOrdersExported($ids);
					if ($orders === []) {
						$control->presenter->flashMessageSuccess($this->translator->translate('ceskaposta.labels.noOrders'));
						$control->presenter->redrawControl('flashes');

						return;
					}

					set_time_limit(120);
					$result = $this->apiService->generateLabels($orders);

					if ($result['error'] === 0) {
						foreach ($result['files'] as $file) {
							$arr                                         = [
								'name' => 'Ceska posta labels',
								'url'  => $control->presenter->link(':Core:Admin:FileResponse:', base64_encode(serialize([
									'file'     => $file,
									'filename' => basename($file),
								]))),
							];
							$control->presenter->payload->fileRequests[] = $arr;
						}

						$control->presenter->flashMessageSuccess($this->translator->translate('ceskaposta.labels.success'));
					} else {
						$control->presenter
							->flashMessageDanger($this->translator->translate('ceskaposta.labels.error'));
					}

					$control->presenter->redrawControl('flashes');
				});
			}
		}
	}

	public function ordersDefaultNavigationCallback(OrderNavigationCallbackEvent $event): void
	{
		if ($event->key !== 'ceskapostaExport') {
			return;
		}

		$orders = $this->ordersExported->getOrdersNotExported();

		$ids = [];
		foreach ($orders as $k => $order) {
			$ids[] = $order->getOrder()->getId();
		}

		if ($ids !== []) {
			$allowed = $this->speditionsService->checkOrdersGridMenuExportOnlyInStatus($ids);

			foreach ($orders as $k => $order) {
				if (!in_array($order->getOrder()->getId(), $allowed)) {
					unset($orders[$k]);
				}
			}
		}

		if ($orders === []) {
			$event->presenter->flashMessageSuccess($this->translator->translate('ceskaposta.export.noOrders'));
			$event->presenter->redrawControl('flashes');

			return;
		}

		set_time_limit(120);
		$result = $this->apiService->sendOrders($orders);

		if (isset($result['message']) && is_array($result['message'])) {
			foreach ($result['message'] as $messages) {
				if (!is_array($messages)) {
					continue;
				}
				foreach ($messages as $k => $m) {
					match ($k) {
						'danger' => $event->presenter->flashMessageDanger($m),
						'warning' => $event->presenter->flashMessageWarning($m),
						default => $event->presenter->flashMessageInfo($m),
					};
				}
			}
		}

		if ($result['error'] === 0) {
			$event->presenter->flashMessageSuccess($this->translator->translate('ceskaposta.export.success'));
		} else if ($result['error'] > 0 && $result['ok'] > 0) {
			$event->presenter->flashMessageWarning($this->translator->translate('ceskaposta.export.partial', null, $result));
		} else {
			$event->presenter->flashMessageDanger($this->translator->translate('ceskaposta.export.error'));
		}

		$event->presenter->redrawControl('flashes');
	}

	public function expeditionAdvancedOptionsFormSuccess(FormSuccessEvent $event): void
	{
		$presenter = $event->presenter;
		$orders    = $this->ordersExported->getOrdersNotExported($event->values->orderIds);
		if ($orders === []) {
			$presenter->flashMessageSuccess($this->translator->translate('ceskaposta.export.noOrders'));
			$presenter->redrawControl('flashes');

			return;
		}

		$quantity = $event->values->quantity;
		set_time_limit(120);

		if (((int) $quantity) > 1 && Arrays::some($orders, static fn($order): bool => $order instanceof PostWarehouseOrder)) {
			$event->form->addError($this->translator->translate('ceskaposta.export.multiplePiecesOrderWarehouse'));
			$presenter->flashMessageWarning($this->translator->translate('ceskaposta.export.multiplePiecesOrderWarehouse'));
			$presenter->redrawControl('flashes');

			return;
		}
		$result = $this->apiService->sendOrders($orders, max(1, (int) $quantity), (array) $event->form->getHttpData()['advancedOptions']);

		if (isset($result['message']) && is_array($result['message'])) {
			foreach ($result['message'] as $messages) {
				if (!is_array($messages)) {
					continue;
				}
				foreach ($messages as $k => $m) {
					match ($k) {
						'danger' => $presenter->flashMessageDanger($m),
						'warning' => $presenter->flashMessageWarning($m),
						default => $presenter->flashMessageInfo($m),
					};
				}
			}
		}

		if ($result['error'] === 0) {
			$presenter->flashMessageSuccess($this->translator->translate('ceskaposta.export.success'));
		} else if ($result['error'] > 0 && $result['ok'] > 0) {
			$presenter->flashMessageWarning($this->translator->translate('ceskaposta.export.partial', null, $result));
		} else {
			$presenter->flashMessageDanger(is_string($result['message']) && strlen($result['message']) > 0 ? $result['message'] : $this->translator->translate('ceskaposta.export.error'));
		}

		$presenter->redrawControl('flashes');
	}

}
