<?php declare(strict_types = 1);

namespace Ceskaposta\AdminModule\Model\Subscribers;

use Ceskaposta\Model\CeskaPostaConfig;
use Ceskaposta\Model\Libs\CeskaPostaApi;
use Ceskaposta\Model\OrdersExported;
use Contributte\Translation\Translator;
use Core\Model\Event\Event;
use EshopOrders\AdminModule\Model\Dao\OrderExpedition;
use EshopOrders\AdminModule\Model\Orders;
use Override;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class ExpeditionSubscriber implements EventSubscriberInterface
{
	public function __construct(
		protected CeskaPostaApi  $apiService,
		protected OrdersExported $ordersExported,
		protected Translator     $translator,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			Orders::class . '::processOrdersForExpedition' => 'processOrdersForExpedition',
			Orders::class . '::sendExpeditionData'         => 'sendExpeditionData',
			Orders::class . '::generateLabel'              => 'generateLabel',
		];
	}

	public function processOrdersForExpedition(Event $event): void
	{
		if (!CeskaPostaConfig::load('allowGenerateLabel', false)) {
			return;
		}

		$orderExpeditions = [];
		$ids              = $this->filterOrders($event->data['data']);

		if ($ids !== []) {
			foreach ($this->ordersExported->getOrdersNotExported($ids) as $id => $expedition) {
				$orderExpeditions[$id] = new OrderExpedition($expedition->getOrder(), $this->translator->translate('ceskaposta.name'), $expedition);
			}

			$event->data['expedition'] += $orderExpeditions;
		}
	}

	public function sendExpeditionData(Event $event): void
	{
		$ids             = $this->filterOrders($event->data['data']);
		$quantity        = $event->data['quantity'] ?? 1;
		$advancedOptions = $event->data['advancedOptions'] ?? [];

		if ($quantity > (int) CeskaPostaConfig::load('maxPackages')) {
			$event->data['okCount']                       = 0;
			$event->data['errorCount']                    = 1;
			$event->data['message']['warning']['warning'] = $this->translator->translate('ceskaposta.export.maxPackages', ['quantity' => CeskaPostaConfig::load('maxPackages')]);

			return;
		}

		if ($ids !== []) {
			$orders = $this->ordersExported->getOrdersNotExported($ids);

			foreach ($orders as $o) {
				$result                    = $this->apiService->sendOrders([$o], $quantity, $advancedOptions);
				$event->data['okCount']    += $result['ok'];
				$event->data['errorCount'] += $result['error'];

				if (isset($result['message'])) {
					$event->data['message'][$result['messageClass'] ?? 'info'][] = $result['message'];
				}
			}
		}
	}

	public function generateLabel(Event $event): void
	{
		$ids = $this->filterOrders($event->data['data']);

		if ($ids !== []) {
			$orders = $this->ordersExported->getOrdersExported($ids);

			if ($orders !== []) {
				$result = $this->apiService->generateLabels($orders);

				if ($result['error'] === 0) {
					foreach ($result['files'] as $file) {
						$event->data['files'][] = $file;
					}
					$event->data['okCount'] += $result['ok'];
				} else {
					$event->data['errorCount'] += $result['error'];
				}
			}
		}
	}

	protected function filterOrders(array $data): array
	{
		$ids = [];
		foreach ($data as $v) {
			if (\str_starts_with((string) $v['service'], 'cpost')) {
				$ids[] = $v['id'];
			}
		}

		return $ids;
	}

}
