<?php declare(strict_types = 1);

namespace Ceskaposta\Model\Subscribers;

use Ceskaposta\Model\CeskaPostaConfig;
use Ceskaposta\Model\Entities\ParcelDeliveryToHandOrder;
use Ceskaposta\Model\Entities\PostOfficeOrder;
use Ceskaposta\Model\Entities\PostWarehouseOrder;
use Ceskaposta\Model\ParcelsService;
use Core\Model\Countries;
use Core\Model\Entities\EntityManagerDecorator;
use EshopOrders\FrontModule\Model\Dao\PaymentSpedition;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Event\OrderFormEvent;
use EshopOrders\FrontModule\Model\Event\SaveOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Event\SetOrderFormDataEvent;
use EshopOrders\Model\PaymentSpeditions;
use Nette\ComponentModel\IComponent;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;
use Override;
use stdClass;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderFormSubscriber implements EventSubscriberInterface
{
	public function __construct(
		protected EntityManagerDecorator $em,
		protected Request                $httpRequest,
		protected PaymentSpeditions      $paymentSpeditions,
		protected ParcelsService         $parcelsService,
		protected Countries              $countriesService,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.createOrderForm'        => ['createOrderForm', 100],
			'eshopOrders.setOrderFormData'       => ['setOrderFormData', 100],
			'eshopOrders.orderFormValidate'      => ['orderFormValidate', 100],
			'eshopOrders.orderBeforeSave'        => ['orderBeforeSave', 100],
			'eshopOrders.saveOrderFormDataStep2' => ['saveOrderFormDataStep2', 100],
			'eshopOrders.saveOrderFormDataStep3' => ['saveOrderFormDataStep3', 100],
		];
	}

	public function createOrderForm(OrderFormEvent $event): void
	{
		$form = $event->form;

		$templates   = $form->getCustomData('speditionServiceTemplates') ?: [];
		$templates[] = __DIR__ . '/speditionTemplate.latte';
		$form->addCustomData('speditionServiceTemplates', $templates);

		$templates   = $form->getCustomData('speditionSummaryTemplates') ?: [];
		$templates[] = __DIR__ . '/speditionSummaryTemplate.latte';
		$form->addCustomData('speditionSummaryTemplates', $templates);

		$container = $form->addContainer('cpost');

		foreach ($this->getCzechPostSpeditions() as $row) {
			$speditionId = $row->getSpedition()->getId();

			foreach ($row->getCountries() as $country) {
				$k = $country->getId() . '_' . $speditionId;
				if ($container->getComponent($k, false) instanceof IComponent) {
					continue;
				}

				if ($row->getSpedition()->code1 === 'napostu') {
					$container->addHidden($k);
					if (CeskaPostaConfig::load('mapPicker', false)) {
						$container->addText($k . '_p')
							->setPlaceholder('ceskapostaFront.orderForm.selectDispensingPoint')
							->setHtmlAttribute('autocomplete', 'off')
							->setHtmlAttribute('data-type', 'POST_OFFICE')
							->setHtmlAttribute('data-cpost-pickup-trigger', $container->getComponent($k)->getHtmlId());
					} else {
						$container->addText($k . '_p')
							->setPlaceholder('ceskapostaFront.orderForm.postOfficeName')
							->setHtmlAttribute('data-key', $k)
							->getControlPrototype()->class[] = 'cpostNaPostuTrigger';
					}
				} else if ($row->getSpedition()->code1 === 'balikovna') {
					$container->addHidden($k);
					if (CeskaPostaConfig::load('mapPicker', false)) {
						$container->addText($k . '_p')
							->setPlaceholder('ceskapostaFront.orderForm.selectDispensingPoint')
							->setHtmlAttribute('autocomplete', 'off')
							->setHtmlAttribute('readonly')
							->setHtmlAttribute('data-type', 'BALIKOVNY')
							->setHtmlAttribute('data-cpost-pickup-trigger', $container->getComponent($k)->getHtmlId());
					} else {
						$container->addText($k . '_p')
							->setPlaceholder('ceskapostaFront.orderForm.postWarehouseName')
							->setHtmlAttribute('data-key', $k)
							->getControlPrototype()->class[] = 'cpostBalikovnaTrigger';
					}
				}

				$event->template->cpostLastKey = $k;
			}
		}
	}

	public function setOrderFormData(SetOrderFormDataEvent $event): void
	{
		$data = $event->data;
		$form = $event->form;
		/** @var ArrayHash $value */
		$value = $form->getUntrustedValues();

		foreach ($data['cpost'] ?? [] as $k => $v) {
			if (isset($form['cpost'][$k])) {
				$form['cpost'][$k]->setDefaultValue($v);
			}
		}

		$parcel = $this->getParcelByData($data);
		if ($parcel instanceof stdClass) {
			foreach (['company', 'firstName', 'lastName', 'email', 'phone'] as $k) {
				$form->getComponent($k . '2')->setValue($form->getComponent($k . '2')->isRequired() && !$value[$k] ? 'placeholder' : ($value[$k] ?: null));
			}

			$form->getComponent('useAddrDeli')->setValue(true);
			$form->getComponent('street2')->setValue($parcel->c_obce);
			$form->getComponent('city2')->setValue($parcel->obec);
			$form->getComponent('postal2')->setValue(str_replace(' ', '', (string) $parcel->psc));
			$form->getComponent('country2')->setValue($data['speditionCountry']);
		}
	}

	public function saveOrderFormDataStep2(SaveOrderFormDataEvent $event): void
	{
		$data    = &$event->data;
		$request = $this->httpRequest;

		$speditions = $this->getCzechPostSpeditions();
		$spedition  = $speditions[$data['payment'] . '_' . $data['spedition']] ?? null;

		if ($spedition) {
			$k                        = $data['speditionCountry'] . '_' . $data['spedition'];
			$data['cpost'][$k]        = $request->getPost('cpost')[$k];
			$data['cpost'][$k . '_p'] = $request->getPost('cpost')[$k . '_p'];
			if ($spedition->getSpedition()->isPickup) {
				$data['disableDeliveryAddressSpedition'] = 'cpost';
				$data['disableDeliveryAddress']          = true;
				$data['useAddrDeli']                     = false;
			} else if ($data['disableDeliveryAddressSpedition'] == 'cpost') {
				unset($data['disableDeliveryAddress']);
			}
		} else if ($data['disableDeliveryAddressSpedition'] == 'cpost') {
			unset($data['disableDeliveryAddress']);
		}
	}

	public function saveOrderFormDataStep3(SaveOrderFormDataEvent $event): void
	{
		$data = &$event->data;
		$form = &$event->form;

		$parcel = $this->getParcelByData($data);

		if ($parcel instanceof stdClass) {
			$d = [];
			foreach (['company', 'firstName', 'lastName', 'email', 'phone'] as $k) {
				$d[$k . '2'] = $data[$k];
			}

			$countryId   = $data['speditionCountry'];
			$countryText = $this->countriesService->getAllNameColumn()[$countryId] ?? $countryId;

			$d += [
				'useAddrDeli'  => true,
				'street2'      => $parcel->c_obce,
				'city2'        => $parcel->obec,
				'postal2'      => str_replace(' ', '', (string) $parcel->psc),
				'country2'     => $countryId,
				'country2Text' => $countryText,
			];

			$data = array_merge($data, $d);
			$form->setValues($d);
		}
	}

	public function orderFormValidate(OrderFormEvent $event): void
	{
		$form = $event->form;
		/** @var ArrayHash $values */
		$values = $form->getUntrustedValues();
		$psKey  = $values->payments->{$values->speditionCountry} . '_' . $values->speditions->{$values->speditionCountry};

		$spedition = $this->getCzechPostSpeditions()[$psKey] ?? null;

		if ($spedition) {
			$branchId = $values->cpost->{$values->speditionCountry . '_' . $spedition->getSpedition()->getId()};

			if (!$branchId) {
				if ($spedition->getSpedition()->code1 === 'napostu') {
					$form->addError('ceskapostaFront.orderForm.postOfficePscMissing');
				} else if ($spedition->getSpedition()->code1 === 'balikovna') {
					$form->addError('ceskapostaFront.orderForm.postWarehousePscMissing');
				}
			}
		}
	}

	public function orderBeforeSave(OrderEvent $event): void
	{
		$values = $event->formData;

		$spedition = $this->getCzechPostSpeditions()[$values->payments->{$values->speditionCountry} . '_' . $values->speditions->{$values->speditionCountry}] ?? null;

		if (!$spedition && $event->orderSpedition && $event->orderSpedition->getSpedition()->getIdent() === 'cpost') {
			$spedition = $event->orderSpedition;
		}

		if ($spedition) {
			if ($spedition->getSpedition()->isPickup) {
				$branchId = $values->cpost->{$values->speditionCountry . '_' . $spedition->getSpedition()->getId()};
				$parcel   = $this->parcelsService->getParcelByService($branchId, $spedition->getSpedition()->code1);
				if ($parcel instanceof stdClass) {
					if ($spedition->getSpedition()->code1 === 'napostu') {
						$entity           = new PostOfficeOrder($event->order, $branchId);
						$entity->postName = $parcel->naz_prov;
					} else if ($spedition->getSpedition()->code1 === 'balikovna') {
						$entity           = new PostWarehouseOrder($event->order, $branchId);
						$entity->postName = $parcel->nazev;
					} else {
						return;
					}

					$entity->village     = $parcel->obec;
					$entity->partVillage = $parcel->c_obce;

					$this->em->persist($entity);
				}
			} else if ($spedition->getSpedition()->code1 === 'doruky') {
				$entity = new ParcelDeliveryToHandOrder($event->order);
				$this->em->persist($entity);
			}
		}
	}

	/**
	 * @return PaymentSpedition[]
	 */
	protected function getCzechPostSpeditions(): array
	{
		$result = [];
		foreach ($this->paymentSpeditions->getAllPublished() as $row) {
			if ($row->getSpedition()->getIdent() === 'cpost') {
				$result[$row->getPayment()->getId() . '_' . $row->getSpedition()->getId()] = $row;
			}
		}

		return $result;
	}

	public function getParcelByData(array $data): ?stdClass
	{
		$speditions       = $this->getCzechPostSpeditions();
		$paymentSpedition = $speditions[$data['payment'] . '_' . $data['spedition']] ?? null;

		if ($paymentSpedition) {
			$spedition = $paymentSpedition->getSpedition();

			if ($spedition->isPickup) {
				$parcelId = $data['cpost'][$data['speditionCountry'] . '_' . $spedition->getId()] ?? null;

				if ($parcelId) {
					return $this->parcelsService->getParcelByService($parcelId, $spedition->code1);
				}
			}
		}

		return null;
	}
}
