<?php declare(strict_types = 1);

namespace ComGate\Model;

use ComGate\Entity\PaymentStatus;
use ComGate\Gateway\PaymentService;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Sites;
use EshopOrders\FrontModule\Model\CardsPaymentService;
use EshopOrders\Model\Entities\OrderCardPaymentHistory;

class Payments
{
	protected EntityManagerDecorator $em;
	protected PaymentService         $gatewayPaymentService;
	protected CardsPaymentService    $cardsPaymentService;
	protected Sites                  $sites;

	public function __construct(
		EntityManagerDecorator $em,
		PaymentService         $gatewayPaymentService,
		CardsPaymentService    $cardsPaymentService,
		Sites                  $sites
	)
	{
		$this->em                    = $em;
		$this->gatewayPaymentService = $gatewayPaymentService;
		$this->cardsPaymentService   = $cardsPaymentService;
		$this->sites                 = $sites;
	}

	public function checkPaidStatus(): void
	{
		ini_set('memory_limit', '2G');
		if (php_sapi_name() === 'cli' && !ComGateConfig::load('scheduler.allowCheckPaidStatus', true)) {
			return;
		}

		try {
			foreach ($this->getNotCompletePayments() as $row) {
				Sites::$currentIdentOverride = $row['site_id'];
				Sites::$currentLangOverride  = $row['lang'];
				$complete                    = false;
				$currentSite                 = $this->sites->getCurrentSite();
				if (!$currentSite) {
					continue;
				}

				foreach ($currentSite->getDomains() as $domain) {
					$currentSite->currentLang = $domain->getLang();

					$complete = $this->processComplete($row);

					if ($complete) {
						break;
					}
				}

				if (!$complete) {
					Sites::$currentIdentOverride = ComGateConfig::load('defaultSite');

					foreach ($currentSite->getDomains() as $domain) {
						$currentSite->currentLang = $domain->getLang();

						$complete = $this->processComplete($row);

						if ($complete) {
							break;
						}
					}
				}
			}
		} catch (\Exception $e) {
		}
	}

	protected function processComplete(array $row): bool
	{
		$response = $this->gatewayPaymentService->status(PaymentStatus::of($row['param']));

		if ($response->isOk()) {
			$data = $response->getData();

			if ($data['status'] === 'CANCELLED') {
				$this->cardsPaymentService->tokenCancelled($row['token_id']);
			} else if ($data['status'] === 'PAID') {
				$this->cardsPaymentService->tokenPaid($row['token_id']);
				if ($this->em->getConnection()->isTransactionActive()) {
					$this->em->commit();
				}
			}

			return true;
		}

		return false;
	}

	public function getNotCompletePayments(): array
	{
		$tokens = [];

		// Získání zakladních dat
		foreach ($this->em->getConnection()->fetchAllAssociative("SELECT
					t.order_id, t.created, t.param, t.id as token_id, h1.status, o.site_id, o.lang
				FROM
					eshop_orders__order_card_payment_token t
				    INNER JOIN eshop_orders__order o ON (o.id = t.order_id AND o.is_paid != 1)
					JOIN eshop_orders__order_card_payment_history h1 ON ( t.id = h1.token_id )
					LEFT OUTER JOIN eshop_orders__order_card_payment_history h2 ON (
						t.id = h2.token_id 
						AND ( h1.created < h2.created OR ( h1.created = h2.created AND h1.id < h2.id ) ) 
					) 
				WHERE
					h2.id IS NULL
					AND t.param IS NOT NULL
				HAVING
					h1.status = 'checkout'") as $row) {
			$tokens[$row['token_id']] = $row;
		}

		// Kontrola dvojitých stavů
		foreach ($this->em->createQueryBuilder()->select('ph.id, IDENTITY(ph.token) as token')
			         ->from(OrderCardPaymentHistory::class, 'ph')
			         ->where('ph.token IN (:ids)')
			         ->andWhere('ph.status IN (:statuses)')
			         ->setParameters([
				         'ids'      => array_keys($tokens),
				         'statuses' => [
					         OrderCardPaymentHistory::STATUS_PAID,
					         OrderCardPaymentHistory::STATUS_CANCELLED,
				         ],
			         ])
			         ->getQuery()->getArrayResult() as $row) {
			unset($tokens[$row['token']]);
		}

		return $tokens;
	}
}
