<?php declare(strict_types = 1);

namespace ComGate\FrontModule\Model\Subscribers;

use Brick\Money\Money;
use ComGate\Comgate;
use ComGate\Entity\AbstractPayment;
use ComGate\Entity\Payment;
use ComGate\Entity\PaymentStatus;
use ComGate\Entity\Storno;
use ComGate\FrontModule\Components\ComgateControl;
use ComGate\FrontModule\Components\IComgateControlFactory;
use ComGate\FrontModule\Model\Exceptions\ComgateException;
use ComGate\Gateway\PaymentService;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\PresenterTemplateEvent;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Strings;
use EshopOrders\FrontModule\Model\CardsPaymentService;
use EshopOrders\FrontModule\Presenters\FinishedPresenter;
use EshopOrders\FrontModule\Presenters\PaymentPresenter;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Orders;
use Nette\Utils\Json;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class OrderPaymentSubscriber implements EventSubscriberInterface
{
	protected IComgateControlFactory $comgateControlFactory;
	protected PaymentService         $paymentService;
	protected CardsPaymentService    $cardsPaymentService;
	protected EntityManagerDecorator $em;
	protected Orders                 $orders;

	public function __construct(
		IComgateControlFactory $comgateControlFactory,
		PaymentService         $paymentService,
		CardsPaymentService    $cardsPaymentService,
		Orders                 $orders,
		EntityManagerDecorator $em
	)
	{
		$this->comgateControlFactory = $comgateControlFactory;
		$this->paymentService        = $paymentService;
		$this->cardsPaymentService   = $cardsPaymentService;
		$this->orders                = $orders;
		$this->em                    = $em;
	}

	/**
	 * @return string[]
	 */
	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.orderFinishedRender'   => 'orderFinishedRender',
			'eshopOrders.paymentPayAction'      => 'paymentPayAction',
			'eshopOrders.paymentFinishedRender' => 'paymentFinishedRender',
		];
	}

	/**
	 * @param PresenterTemplateEvent $event
	 */
	public function orderFinishedRender(PresenterTemplateEvent $event): void
	{
		/** @var FinishedPresenter $presenter */
		$presenter = $event->presenter;
		$order     = $presenter->order;

		if (!$order || !$order->getIdent() || !$this->checkPaymentType($order)) {
			return;
		}

		$presenter->template->link = $presenter->link('Payment:pay', ['orderIdent' => $order->getIdent(), 'activeNavigation' => null]);
		$presenter->addIncludeTemplate(__DIR__ . '/ComgateButton.latte');
	}

	public function paymentPayAction(PresenterTemplateEvent $event): void
	{
		/** @var PaymentPresenter $presenter */
		$presenter = $event->presenter;
		$order     = $presenter->order;
		$self      = $this;

		if (!$order || !$order->getId() || !$this->checkPaymentType($order)) {
			return;
		}

		$token = $this->cardsPaymentService->getLastCreated($order->getId()) ?: $this->cardsPaymentService->createToken($order);

		if ($token->param) {
			$statusResponse = $this->paymentService->status(PaymentStatus::of($token->param));
			$status         = $statusResponse->getData();

			if ($status['status'] !== 'CANCELLED') {
				if (!$status['curr']) {
					$status['curr'] = $order->getCurrencyCode();
				}

				$orderMoney  = Money::of(round($order->getPrice(true), 2), $order->getCurrencyCode());
				$statusMoney = Money::of(round($status['price'] / 100, 2), $status['curr']);

				if ($orderMoney->getCurrency()->is($status['curr']) && $orderMoney->getAmount()->isEqualTo($statusMoney->getAmount())) {
					header("Location: " . Comgate::getPaymentUrl($token->param));
					exit;
				}
			}

			$this->em->beginTransaction();

			$stornoResponse = $this->paymentService->storno(Storno::of($token->param));
			$stornoData     = $stornoResponse->getData();

			if ($stornoData['code'] === '0') {
				$self->cardsPaymentService->tokenCancelled((int) $token->getId(), 'Invalid price or currency');
			}

			$token = $this->cardsPaymentService->createToken($order);
			$this->em->commit();
		}

		$email = $order->getAddressInvoice() ? $order->getAddressInvoice()->getEmail() : null;
		if (!$email && $order->getAddressDelivery()) {
			$email = $order->getAddressDelivery()->getEmail();
		}

		if (!$email && $order->getCustomer()) {
			$email = $order->getCustomer()->getUser()->getEmail();
		}

		switch ($order->getPayment()->payment->code1 ?? null) {
			case 'laterOnline':
				$method = 'LATER_ALL';
				break;
			case 'laterTwistoOnline':
				$method = 'LATER_TWISTO';
				break;
			case 'partOnline':
				$method = 'PART_ALL';
				break;
			case 'partTwistoOnline':
				$method = 'PART_TWISTO';
				break;
			case 'loanOnline':
				$method = 'LOAN_COFIDIS';
				break;
			case 'googlePay':
				$method = 'GOOGLEPAY_REDIRECT';
				break;
			case 'applePay':
				$method = 'APPLEPAY_REDIRECT';
				break;
			case 'PAYPAL':
				$method = 'PAYPAL';
				break;
			default:
				$method = 'CARD_CZ_CSOB_2';
				break;
		}

		$productsNames = '';
		foreach ($order->getOrderItems() as $orderItem) {
			$text = $orderItem->getOrderItemText();

			if ($text) {
				$productsNames .= ', ' . $text->getName();
			}
		}

		$payment = Payment::of(
			Money::of(round($order->getPrice(true), 2), $order->getCurrencyCode()),
			$productsNames,
			(string) $order->getId(),
			(string) $email,
			$method
		);

		$paymentLang = Arrays::contains(['cs', 'sk', 'en', 'es', 'it', 'pl', 'fr', 'ro', 'de', 'hu', 'si', 'hr', 'no', 'sv'], $order->lang)
			? $order->lang
			: 'cs';

		$payment->setLang($paymentLang);

		if ($productsNames !== '') {
			$payment->setName($productsNames);
		}

		$addrInv = $order->getAddressInvoice() ?: null;
		if ($addrInv) {
			$countryCode = $addrInv->getCountry() ? Strings::upper((string) $addrInv->getCountry()->getId()) : null;

			$payment->fullName              = $addrInv->getName() ?: null;
			$payment->phone                 = $addrInv->getPhone()
				? str_replace(' ', '', Strings::phoneFormat($addrInv->getPhone(), $countryCode, false))
				: null;
			$payment->billingAddrCity       = $addrInv->getCity() ?: null;
			$payment->billingAddrStreet     = $addrInv->getStreet() ?: null;
			$payment->billingAddrCountry    = $countryCode;
			$payment->billingAddrPostalCode = $addrInv->getPostal() ?: null;
		}

		if ($order->getSpeditionIdent() === 'online') {
			$payment->setDelivery(AbstractPayment::deliveryElectronic);
		} else if ($order->getSpedition() && $order->getSpedition()->getSpedition() && $order->getSpedition()->getSpedition()->isPickup) {
			$payment->setDelivery(AbstractPayment::deliveryPickup);
		} else {
			$addrDel = $order->getAddressDelivery() ?: null;

			$payment->setDelivery(AbstractPayment::deliveryHome);

			if ($addrDel) {
				$payment->homeDeliveryCity       = $addrDel->getCity() ?: null;
				$payment->homeDeliveryStreet     = $addrDel->getStreet() ?: null;
				$payment->homeDeliveryCountry    = $addrDel->getCountry() ? Strings::upper((string) $addrDel->getCountry()->getId()) : null;
				$payment->homeDeliveryPostalCode = $addrDel->getPostal() ?: null;
			}
		}

		$control = $this->comgateControlFactory->create($payment);

		$presenter->addComponent($control, 'comGateButton');

		$control->onCheckout[] = static function(ComgateControl $control, Payment $payment, $data) use ($token, $self) {
			$token->param = $data['transId'];
			$self->em->persist($token);
			$self->em->flush($token);
			$self->cardsPaymentService->tokenCheckout((int) $token->getId());

			if ($self->em->getConnection()->isTransactionActive()) {
				$self->em->commit();
			}
		};

		$control->handleCheckout();
	}

	/**
	 * @param PresenterTemplateEvent $event
	 */
	public function paymentFinishedRender(PresenterTemplateEvent $event): void
	{
		/** @var PaymentPresenter $presenter */
		$presenter = $event->presenter;
		$self      = $this;
		$params    = $presenter->getParameters();
		$transId   = $params['transId'] ?: $presenter->getHttpRequest()->getPost('transId');
		$orderId   = $params['orderId'] ?: $presenter->getHttpRequest()->getPost('orderId');

		if (!$orderId) {
			$orderId = $params['refId'] ?: $presenter->getHttpRequest()->getPost('refId');
		}

		$status = null;
		$msg    = null;
		Debugger::log(Json::encode((array) $presenter->getHttpRequest()->getPost()), '_comgate-full-post');

		if (!$transId) {
			try {
				$input = (string) file_get_contents('php://input');
				Debugger::log($input, '_comgate-full-php-input');
				parse_str($input, $res);

				$transId = (int) $res['transId'];
				$orderId = $res['refId'] ?? null;
				$status  = $res['status'] ?? null;
				$msg     = $res['message'] ?? null;
				Debugger::log($res, '_comgate-full-php-input');
			} catch (\Exception $e) {
				Debugger::log($e, 'comgate-php-input');
			}
		}

		if ($transId) {
			$order = $this->orders->get((int) $orderId);

			if (!$order || !$order->getId() || !$this->checkPaymentType($order)) {
				return;
			}

			$presenter->order           = $order;
			$presenter->template->order = $order;

			$response = !$status ? $this->paymentService->status(PaymentStatus::of($transId)) : null;
			$token    = $this->cardsPaymentService->getLastCreated($order->getId());

			if ($response) {
				try {
					Debugger::log(Json::encode($response->getData()), '_comgate-full-response');
				} catch (\Exception$e) {
					Debugger::log($e, '_comgate-full-response-error');
					Debugger::log($response, '_comgate-full-response-error');
				}
			}

			if ($status === 'PAID' || ($response && $response->isOk() && $response->getData()['status'] === 'PAID')) {
				if ($token) {
					$self->cardsPaymentService->tokenPaid($token->getToken());
				}
			} else if ($status === 'PENDING' || ($response && $response->isOk() && $response->getData()['status'] === 'PENDING')) {
				if ($token) {
					$self->cardsPaymentService->tokenPending($token->getToken());
				}
			} else {
				if (!$msg && $response) {
					$msg = $response->getData()['message'] ?? null;
				}

				Debugger::log(new ComgateException($msg ?? ''), 'comgate');
				if (!$status && $token) {
					$self->cardsPaymentService->tokenError($token->getToken(), (string) $msg);
				}

				$presenter->template->cardError = $msg ?: $status;
			}

			$presenter->addIncludeTemplate(__DIR__ . '/PaymentFinished.latte');
		}
	}

	private function checkPaymentType(?Order $order): bool
	{
		if (
			!$order
			|| $order->getPaymentIdent() !== 'card'
			|| ($order->getPayment() && $order->getPayment()->getPayment() && $order->getPayment()->getPayment()->code1 === 'paypal')
		) {
			return false;
		}

		return true;
	}

}
