<?php declare(strict_types = 1);

namespace Core\Console\Database;

use Doctrine;
use Doctrine\DBAL\Migrations\Provider\SchemaProviderInterface;
use Doctrine\DBAL\Migrations\Configuration\Configuration;
use Doctrine\DBAL\Migrations\Tools\Console\Command\MigrateCommand;
use Kdyby;
use Kdyby\Doctrine\EntityManager;
use Nette\Neon\Neon;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class Migrate extends MigrateCommand
{
	/** @var EntityManager */
	public $em;

	/** @var EntityManager */
	protected $entityManager;

	/** @var SchemaProviderInterface */
	protected $schemaProvider;

	/** @var string */
	protected $extensionName;

	public function __construct(EntityManager $em, $name = null)
	{
		parent::__construct($name);
		$this->em = $em;
	}

	protected function configure()
	{
		$this->setName('database:migrate')->setDescription('Zmigrování změn DB')
			->setDescription('Spustí migraci DB');
		$this->addOption('db-configuration', null, InputOption::VALUE_OPTIONAL, 'The path to a database connection configuration file.');
		$this->addOption('configuration', null, InputOption::VALUE_OPTIONAL, 'The path to a migrations configuration file.');
	}

	public function execute(InputInterface $input, OutputInterface $output)
	{
		$configuration = $this->getMigrationConfiguration($input, $output);
		$migration     = $this->createMigration($configuration);

		$this->outputHeader($configuration, $output);

		$executedMigrations  = $configuration->getMigratedVersions();
		$availableMigrations = $configuration->getAvailableVersions();

		$executedUnavailableMigrations = array_diff($executedMigrations, $availableMigrations);
		if (!empty($executedUnavailableMigrations)) {
			$output->writeln(sprintf(
				'<error>WARNING! You have %s previously executed migrations'
				. ' in the database that are not registered migrations.</error>',
				count($executedUnavailableMigrations)
			));

			foreach ($executedUnavailableMigrations as $executedUnavailableMigration) {
				$output->writeln(sprintf(
					'    <comment>>></comment> %s (<comment>%s</comment>)',
					$configuration->getDateTime($executedUnavailableMigration),
					$executedUnavailableMigration
				));
			}
		}

		$cancelled = false;
		$migration->migrate();
	}

	private function getEntityManager()
	{
		if (!$this->entityManager) {
			$extensionName       = strtolower($this->extensionName);
			$this->entityManager = (clone $this->em);
			$this->entityManager->getConfiguration()
				->setEntityNamespaces([$extensionName . '\Model\Entities' => SRC_DIR . '/' . $extensionName . '/src/Model/Entities']);

			$driver = new Doctrine\ORM\Mapping\Driver\AnnotationDriver(new Doctrine\Common\Annotations\AnnotationReader());
			$driver->addPaths([SRC_DIR . '/' . $extensionName . '/src/Model/Entities']);
			$this->entityManager->getConfiguration()->setMetadataDriverImpl($driver);
		}

		return $this->entityManager;
	}

	/**
	 * Konfigurace a načtení migračních souborů pro aktuální nastavení aplikace
	 *
	 * @param InputInterface  $input
	 * @param OutputInterface $output
	 *
	 * @return Configuration
	 */
	protected function getMigrationConfiguration(InputInterface $input, OutputInterface $output)
	{
		$configuration = parent::getMigrationConfiguration($input, $output);

		// Načtení aktivních rozšíženích ze systému
		$neon = Neon::decode(file_get_contents(APP_DIR . '/config/app/cms.neon'));
		foreach ($neon['extensions'] as $name => $class) {
			$dir = SRC_DIR . '/' . strtolower($name) . '/src/migrations';
			if (file_exists($dir))
				$configuration->registerMigrationsFromDirectory($dir);
		}

		// Načtení z rozšíženích pro aplikaci
		foreach (glob(CUSTOM_DIR . '/*/migrations', GLOB_ONLYDIR) as $dir) {
			$configuration->registerMigrationsFromDirectory($dir);
		}

		return $configuration;
	}
}