<?php declare(strict_types = 1);

namespace Core\FrontModule\Presenters;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Components\Navigation\INavigationFactory;
use Core\FrontModule\Components\IContactFormFactory;
use Core\FrontModule\Components\StructureData\IDataSchemaFactory;
use Core\FrontModule\Model\SeoParser;
use Google\FrontModule\Components\IGtmRemarketingFactory;
use Google\FrontModule\Components\IGtmUserTrackingFactory;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Navigations;
use Users\Model\UsersActivity;

/**
 * Class BasePresenter
 * @package Core\FrontModule\Presenters
 */
abstract class BasePresenter extends CustomBasePresenter
{
	/** @var Navigations @inject */
	public $navigationsService;

	/** @var SeoParser @inject */
	public $seoParserService;

	/** @var UsersActivity @inject */
	public $usersActivity;

	/** @var Navigation */
	private $activeNavigation;

	/** @var DaoNavigationItem */
	protected $homepage;

	/** @var string */
	public $baseUrl;

	/** @var array */
	public $seoDataFromEntity = [];

	protected $skipSetInitial = false;

	protected function startup()
	{
		parent::startup();

		if (!$this->skipSetInitial) {
			/** @var DaoNavigationItem $activeNavigation */
			$this->activeNavigation = $activeNavigation = $this->request->getParameter('activeNavigation');
			$this->baseUrl          = $this->getHttpRequest()->getUrl()->getBaseUrl();
		}

		$this->template->pageClass = '';

		if ($this->user->isLoggedIn()) {
			$this->usersActivity->setLastActivity($this->user->getId());
		}
	}

	public function beforeRender()
	{
		parent::beforeRender();

		/** @var DaoNavigationItem $activeNavigation */
		$activeNavigation            = $this->getParameter('activeNavigation');
		$activeNavigation->isCurrent = true;

		$this->activeNavigation = $activeNavigation;

		if ($activeNavigation) {
			if (!$this->skipSetInitial) {
				$this->template->breadcrumb       = $this->navigationsService->getPath($activeNavigation->getId());
				$this->template->activeNavigation = $this->activeNavigation ?: null;
				$this->template->pageClass        .= ($activeNavigation ? $activeNavigation->getParam('pageClass') : '') . ' lang-' . $this->template->lang['shortTag'];
			}

			if ($this->seoDataFromEntity)
				$activeNavigation->setSeo(array_merge((array) $activeNavigation->getSeo(), (array) $this->seoDataFromEntity));

			if (!$this['title']->getTitle()) {
				if ($activeNavigation->isHomepage && !$activeNavigation->getSeo('title'))
					$this->setTitle('$siteName', true);
				else if ($activeNavigation->getSeo('title'))
					$this->setTitle($activeNavigation->getSeo('title'), true);
				else
					$this->setTitle($activeNavigation->title, false);
			}

			$metaSeo = ['description', 'robots'];
			foreach ($metaSeo as $v) {
				if (!$activeNavigation->getSeo($v))
					continue;

				$this['meta']->setMeta($v, $activeNavigation->getSeo($v));
			}
		}

		if (!$this->skipSetInitial) {
			//TODO mozno odstranit
			$this->template->_imagePipe = $this->imagePipe;
			//TODO mozno odstranit
			$this->template->lang       = $this->translator->getLocale();
			$this->template->homepage   = $this->getHomepage();
		}

		if ($this->systemConfig->get('testMode') === true)
			$this['meta']->setMeta('robots', 'noindex, nofollow');
	}

	public function getActiveNavigation()
	{
		return $this->activeNavigation;
	}

	/*******************************************************************************************************************
	 * =============================== Handle
	 */

	public function handleLogout()
	{
		$this->getUser()->logout(true);
		$homepage = $this->getHomepage();
		$this->redirectUrl($homepage ? $homepage->link : '/');
	}

	/*******************************************************************************************************************
	 * =============================== Components
	 */

	protected function createComponentContactForm(IContactFormFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentNavigation(INavigationFactory $factory)
	{
		$control = $factory->create();
		$control->setData($this->navigationsService->getPublishedDaoNavigationStructure());

		return $control;
	}

	protected function createComponentDataSchemaControl(IDataSchemaFactory $factory)
	{
		return $factory->create();
	}
	
	protected function createComponentGtmRemarketing(IGtmRemarketingFactory $factory)
	{
		return $factory->create();
	}
	
	protected function createComponentGtmUserTracking(IGtmUserTrackingFactory $factory)
	{
		return $factory->create();
	}

	/*******************************************************************************************************************
	 * =============================== GET / SET
	 */

	/**
	 * @param string $title
	 * @param bool   $includeSiteName
	 */
	public function setTitle($title, $includeSiteName = false)
	{
		$this['title']->setTitle($title, $includeSiteName);

		if ($this->isAjax())
			$this->payload->pageTitle = $this['title']->getFullTitle();
	}

	public function getHomepage(): DaoNavigationItem
	{
		if ($this->homepage === null) {
			$this->homepage = $this->navigationsService->getHomepage($this->locale) ?? null;
		}

		return $this->homepage;
	}
}
