<?php declare(strict_types=1);

namespace Core\AdminModule\Components;

use Core\Model\Entities\StaticTextTranslation;
use Core\Model\StaticTexts;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Exception;
use Nette\Application\UI\Multiplier;
use Nette\ComponentModel\IComponent;
use Nette\Utils\ArrayHash;
use Tracy\Debugger;

class StaticTextsForm extends BaseControl
{
	/** @var IStaticTextsFormFactory */
	protected $staticTextsFormFactory;

	/** @var StaticTexts */
	protected $staticTexts;

	/** @var array */
	protected $messages = [];

	/**
	 * StaticTextsForm constructor.
	 * @param IStaticTextsFormFactory $staticTextsFormFactory
	 * @param StaticTexts $staticTexts
	 */
	public function __construct(IStaticTextsFormFactory $staticTextsFormFactory, StaticTexts $staticTexts)
	{
		parent::__construct();
		$this->staticTextsFormFactory = $staticTextsFormFactory;
		$this->staticTexts = $staticTexts;
	}

	/**
	 * @param IComponent $presenter
	 * @throws Exception
	 */
	protected function attached($presenter): void
	{
		parent::attached($presenter);
		$this->staticTexts->merge();
		$this->messages = $this->staticTexts->getMergedMessages();
	}

	/**
	 * @return Multiplier
	 */
	public function createComponentStaticTextForm(): Multiplier
	{
		return new Multiplier(function ($domain): Multiplier {
			return new Multiplier(function ($locale) use ($domain): BaseForm {
				$messages = $this->messages[$domain][$locale];

				$form = $this->createForm();
				$form->setAjax();

				foreach ($messages as $key => $message) {
					$form->addText(str_replace('.', '_', $key), $key)
						 ->setPlaceholder($message)
						 ->setDefaultValue($message)
						 ->setRequired();
				}

				$form->addCustomData('locale', $locale);
				$form->addCustomData('domain', $domain);
				$form->addSubmit('submit', 'default.save');

				$form->onSuccess[] = [$this, 'onSuccess'];

				return $form;
			});
		});
	}

	/**
	 * @param BaseForm $form
	 * @param ArrayHash $values
	 */
	public function onSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			$this->em->beginTransaction();

			foreach ($values as $key => $value) {
				$key = str_replace('_', '.', $key);
				$locale = $form->getCustomData('locale');
				$staticText = $this->staticTexts->getOneByKeyLocale($key, $locale);

				if ($staticText) {
					$staticText->message = $value;
					$this->em->persist($staticText);
				}
			}

			$this->em->flush();
			$this->em->commit();
			$this->staticTexts->clearCache();

			if (($presenter = $this->getPresenter()) !== null) {
				$presenter->flashMessageSuccess('default.saved');
				$presenter->redrawControl('flashes');
			}

		} catch (Exception $exception) {
			Debugger::log($exception);
			$this->em->rollback();

			if (($presenter = $this->getPresenter()) !== null) {
				$presenter->flashMessageDanger('defeault.removeFailed');
			}
		}
	}

	public function render(): void
	{
		$this->template->messages = $this->messages;
		$this->template->render($this->getTemplateFile());
	}
	
}