<?php declare(strict_types=1);

namespace Core\Model;

use Core\Model\Entities\StaticTextTranslation;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Translation\Utils\Helpers;
use Exception;
use Nette\DI\Container;
use Nette\Localization\ITranslator;
use Nette\Neon\Neon;
use Nette\Utils\FileSystem;

/**
 * Class StaticTexts
 * @package Core\Model
 * @method StaticTextTranslation[] getAll()
 */
class StaticTexts extends BaseEntityService
{
	/** @var string */
	protected $entityClass = StaticTextTranslation::class;

	/** @var ITranslator */
	protected $translator;

	/** @var Container */
	protected $container;

	/**
	 * StaticTexts constructor.
	 * @param ITranslator $translator
	 * @param Container $container
	 */
	public function __construct(ITranslator $translator, Container $container)
	{
		$this->translator = $translator;
		$this->container = $container;
	}

	/**
	 * @param string $domain
	 * @param string $locale
	 * @return array
	 */
	public function getByDomainLocale(string $domain, string $locale): array
	{
		$qb = $this->getEr()->createQueryBuilder('st');
		$qb->select('st.id, st.message')->where('st.id LIKE :key AND st.locale = :locale');
		$qb->setParameters([
			'key'	 => $domain . '%',
			'locale' => $locale
		]);

		return $qb->getQuery()->getResult();
	}

	/**
	 * @param string $key
	 * @param string $locale
	 * @return StaticTextTranslation|null
	 */
	public function getOneByKeyLocale(string $key, string $locale): ?StaticTextTranslation
	{
		return $this->getEr()->findOneBy(['id' => $key, 'locale' => $locale]);
	}

	/**
	 * @return array
	 */
	protected function getResources(): array
	{
		$result = [];
		foreach ($this->translator->getTracyPanel()->getResources() as $locale => $res) {
			foreach ($res as $resource => $domain) {
				if (!Helpers::isDomainAllowed($domain, $this->container->getParameters()['system']['staticTextsTranslation']['domains'])) {
					continue;
				}
				$result[$domain][$locale] = $resource;
			}
		}

		return $result;
	}

	/**
	 * @return array
	 */
	public function getMergedMessages(): array
	{
		return $this->getMessages(function (string $domain, string $locale, array $content) {
			foreach ($this->getByDomainLocale($domain, $locale) as $value) {
				if (isset($content[$value['id']])) {
					$content[$value['id']] = $value['message'];
				}
			}

			return $content;
		});
	}

	/**
	 * @throws Exception
	 */
	public function merge(): void
	{
		$delimiter = '.';

		// loaded neon
		$fileResource = $this->getMessages();

		// loaded db
		$dbResource = [];
		foreach ($this->getAll() as $item) {
			$dbResource[explode($delimiter, $item->id)[0]][$item->locale][$item->id] = $item->message;
		}

		$flattenDbResource = Arrays::flattenWithDottedKeys($dbResource);
		$flattenFileResource = Arrays::flattenWithDottedKeys($fileResource);

		// was added to file, add to db
		foreach (array_diff_key($flattenFileResource, $flattenDbResource) as $key => $val) {
			[$domain, $locale, $id] = explode($delimiter, $key, 3);
			$this->em->persist(new StaticTextTranslation($id, $locale, $val));
		}

		// was deleted from file, remove from db
		foreach (array_diff_key($flattenDbResource, $flattenFileResource) as $key => $val) {
			[$domain, $locale, $id] = explode($delimiter, $key, 3);
			$entity = $this->getOneByKeyLocale($id, $locale);

			if ($entity) {
				$this->em->remove($entity);
			}
		}

		$this->em->flush();
		$this->clearCache();
	}

	/**
	 * @param callable|null $contentDecorator
	 * @return array
	 */
	private function getMessages(callable $contentDecorator = null): array
	{
		$result = [];
		foreach ($this->getResources() as $domain => $resources) {
			foreach ($resources as $locale => $resource) {
				$fileContent = @file_get_contents($resource);
				$fileDecoded = Neon::decode($fileContent);

				if ($fileDecoded) {
					$array = Arrays::flattenWithDottedKeys($fileDecoded, $domain . '.');

					if ($contentDecorator) {
						$array = $contentDecorator($domain, $locale, $array);
					}

					$result[$domain][$locale] = $array;
				}
			}
		}

		return $result;
	}

	public function clearCache(): void
	{
		$cacheDir = $this->translator->getCacheDir();
		if($cacheDir !== null && file_exists($cacheDir)) {
			FileSystem::delete($cacheDir);
		}
	}

}