<?php declare(strict_types = 1);

namespace Core\Model\UI\DataGrid;

use Nette\Application\UI\Form;
use Nette\Forms\Controls\SubmitButton;
use Nette\SmartObject;
use Ublaboo\DataGrid\DataGrid;
use Ublaboo\DataGrid\Exception\DataGridGroupActionException;
use Ublaboo\DataGrid\GroupAction\GroupAction;
use Ublaboo\DataGrid\GroupAction\GroupActionCollection;
use Ublaboo\DataGrid\GroupAction\GroupMultiSelectAction;
use Ublaboo\DataGrid\GroupAction\GroupSelectAction;
use Ublaboo\DataGrid\GroupAction\GroupTextAction;
use Ublaboo\DataGrid\GroupAction\GroupTextareaAction;

class CustomGroupActionCollection extends GroupActionCollection
{
	const ID_ATTRIBUTE_PREFIX = 'group_action_custom_item_';

	/**
	 * @var GroupAction[]
	 */
	protected $groupActionsCustom = [];

	public function __construct(DataGrid $datagrid)
	{
		$this->datagrid = $datagrid;
	}

	/**
	 * Get assambled form
	 *
	 * @param Nette\Forms\Container $container
	 *
	 * @return void
	 */
	public function addToFormContainer($container)
	{
		/** @var Nette\Application\UI\Form $form */
		$form         = $container->lookup('Nette\Application\UI\Form');
		$translator   = $form->getTranslator();
		$main_options = [];

		/**
		 * First foreach for filling "main" select
		 */
		foreach ($this->groupActionsCustom as $id => $action) {
			$main_options[$id] = $action->getTitle();
		}

		$container->addSelect('groupActionCustom', '', $main_options)
			->setPrompt('ublaboo_datagrid.choose');

		/**
		 * Second for creating select for each "sub"-action
		 */
		foreach ($this->groupActionsCustom as $id => $action) {
			$control = null;

			if ($action instanceof GroupSelectAction) {
				if ($action->hasOptions()) {
					if ($action instanceof GroupMultiSelectAction) {
						$control = $container->addMultiSelect($id, '', $action->getOptions());
						$control->setAttribute('data-datagrid-multiselect-id', static::ID_ATTRIBUTE_PREFIX . $id);
						$control->setAttribute('data-style', 'hidden');
						$control->setAttribute('data-selected-icon-check', DataGrid::$icon_prefix . 'check');
					} else {
						$control = $container->addSelect($id, $action->getTitle(), $action->getOptions());
						$control->setPrompt('ublaboo_datagrid.choose');
					}

					$control->setAttribute('id', static::ID_ATTRIBUTE_PREFIX . $id);
					$control->setHtmlAttribute('data-grid-group-action-custom-select', '1');
				}
			} elseif ($action instanceof GroupTextAction) {
				$control = $container->addText($id, '');

				$control->setAttribute('id', static::ID_ATTRIBUTE_PREFIX . $id)
					->addConditionOn($container['groupActionCustom'], Form::EQUAL, $id)
					->setRequired($translator->translate('ublaboo_datagrid.choose_input_required'))
					->endCondition();
			} elseif ($action instanceof GroupTextareaAction) {
				$control = $container->addTextarea($id, '');

				$control->setAttribute('id', static::ID_ATTRIBUTE_PREFIX . $id)
					->addConditionOn($container['groupActionCustom'], Form::EQUAL, $id)
					->setRequired($translator->translate('ublaboo_datagrid.choose_input_required'));
			}

			if ($control) {
				/**
				 * User may set a class to the form control
				 */
				if ($class = $action->getClass()) {
					$control->setAttribute('class', $class);
				}

				/**
				 * User may set additional attribtues to the form control
				 */
				foreach ($action->getAttributes() as $name => $value) {
					$control->setAttribute($name, $value);
				}
			}

			$container->addSubmit('submit_' . $id, 'ublaboo_datagrid.execute')
				->setValidationScope([$container])
				->setAttribute('id', strtolower($this->datagrid->getName()) . 'groupActionCustomSubmit');
		}

		$form->onSubmit[] = [$this, 'submitted'];
	}

	/**
	 * Pass "sub"-form submission forward to custom submit function
	 *
	 * @param Form $form
	 *
	 * @return void
	 */
	public function submitted(Form $form)
	{
		if ($form->isSubmitted() instanceof SubmitButton === false)
			return;

		$btnName = $form->isSubmitted()->getName();
		if (!isset($form['groupActionCustom'][$btnName]))
			return;

		$values = $form->getValues();
		$values = $values['groupActionCustom'];

		$id = explode('_', $btnName)[1] ?? null;

		if ($id === null)
			return;

		$http_ids = $form->getHttpData(Form::DATA_LINE | Form::DATA_KEYS, strtolower($this->datagrid->getName()) . '_group_action_item[]');
		$ids      = array_keys($http_ids);

		$this->groupActionsCustom[$id]->onSelect($ids, $values->{$id} ?? null);

		$form['groupActionCustom'][$id]->setValue(null);
	}

	/**
	 * Add one group action (select box) to collection of actions
	 *
	 * @param string $title
	 * @param array  $options
	 *
	 * @return GroupAction
	 */
	public function addGroupSelectAction($title, $options)
	{
		$id = ($s = sizeof($this->groupActionsCustom)) ? ($s + 1) : 1;

		return $this->groupActionsCustom[$id] = new GroupSelectAction($title, $options);
	}

	/**
	 * Add one group action (multiselect box) to collection of actions
	 *
	 * @param string $title
	 * @param array  $options
	 *
	 * @return GroupAction
	 */
	public function addGroupMultiSelectAction($title, $options)
	{
		$id = ($s = sizeof($this->groupActionsCustom)) ? ($s + 1) : 1;

		return $this->groupActionsCustom[$id] = new GroupMultiSelectAction($title, $options);
	}

	/**
	 * Add one group action (text input) to collection of actions
	 *
	 * @param string $title
	 *
	 * @return GroupAction
	 */
	public function addGroupTextAction($title)
	{
		$id = ($s = sizeof($this->groupActionsCustom)) ? ($s + 1) : 1;

		return $this->groupActionsCustom[$id] = new GroupTextAction($title);
	}

	/**
	 * Add one group action (textarea) to collection of actions
	 *
	 * @param string $title
	 *
	 * @return GroupAction
	 */
	public function addGroupTextareaAction($title)
	{
		$id = ($s = sizeof($this->groupActionsCustom)) ? ($s + 1) : 1;

		return $this->groupActionsCustom[$id] = new GroupTextareaAction($title);
	}

	/**
	 * @param string $title
	 *
	 * @return GroupAction
	 */
	public function getGroupAction($title)
	{
		foreach ($this->groupActionsCustom as $action) {
			if ($action->getTitle() === $title) {
				return $action;
			}
		}

		throw new DataGridGroupActionException("Group action $title does not exist.");
	}
}
