<?php declare(strict_types = 1);

namespace Core\Components\Navigation;

use Core\Model\UI\BaseControl;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;
use Nette\Utils\Arrays;

class Navigation extends BaseControl
{
	/** @var ArrayHash */
	private $data;

	/** @var array */
	public $templates = [];

	/** @var Request */
	protected $httpRequest;

	public function __construct(Request $request)
	{
		$this->data        = new ArrayHash();
		$this->httpRequest = $request;
	}

	public function render($group, $template = 'default')
	{
		if (!isset($this->data->{$group}))
			return;

		/** @var DaoNavigationItem[] $data */
		$data = $this->data->{$group};

		$this->sort($data);
		$this->checkAcl($data);
		$this->checkIsCurrent($data);

		$this->template->data = $data;
		$this->template->setFile(isset($this->templates[$template]) ? $this->templates[$template] : $this->getTemplateFile($template));

		$this->template->render();
	}

	/**
	 * Zjištění jestli uživatel může odkaz vidět
	 *
	 * @param DaoNavigationItem[] $data
	 */
	private function checkAcl(&$data)
	{
		$user = $this->getPresenter()->getUser();

		foreach ($data as $k => $v) {
			$allowed = $v->security ? false : true;

			if (!$allowed)
				foreach ($v->security as $m => $ps) {
					if ($user->isAllowed($m, $ps)) {
						$allowed = true;
						break;
					}
				}

			if (!$allowed)
				unset($data[$k]);

			if ($v->childs) {
				$this->checkAcl($v->childs);
			}
		}
	}

	/**
	 * Zjištění jestli je odkaz aktivní
	 *
	 * @param DaoNavigationItem[] $data
	 */
	private function checkIsCurrent(&$data)
	{
		$presenter = $this->getPresenter();

		$fn = function(&$data) use (&$fn, $presenter) {
			foreach ($data as $nav) {
				if ($nav->isHomepage)
					continue;

				$path      = explode('/', $this->httpRequest->getUrl()->path);
				$isCurrent = false;

				do {
					if (implode('/', $path) == $nav->link) {
						$isCurrent = true;
						break;
					}

					array_pop($path);
				} while (!empty($path));

				if ($isCurrent) {
					$nav->set('isCurrent', true);
				}

				if ($nav->childs) {
					$fn($nav->childs);

					foreach ($nav->childs as $child) {
						if ($child->isCurrent) {
							$nav->set('isCurrent', true);
							break;
						}
					}
				}
			}
		};

		$fn($data);
	}

	public function setData($data)
	{
		if (!is_array($data))
			return;

		foreach ($data as $group => $navs) {

			if (!isset($this->data->$group))
				$this->data->$group = [];

			foreach ($navs as $navName => $nav) {
				$this->data->$group[$navName] = $nav instanceof DaoNavigationItem ? $nav : new DaoNavigationItem($nav);
			}
		}
	}

	private function sort(&$data)
	{
		if (is_array($data)) {
			usort($data, function($a, $b) {
				return $a->order - $b->order;
			});

			foreach ($data as &$v)
				if ($v->childs)
					$this->sort($v->childs);
		}

		return $data;
	}
}
