<?php declare(strict_types = 1);

namespace Core\Model\UI\DataGrid;

use Core\Model\UI\DataGrid\Column;
use Core\Model\UI\Form\BaseForm;
use Nette;
use Ublaboo\DataGrid\DataGrid;
use Ublaboo\DataGrid\InlineEdit\InlineEdit;

class BaseDataGrid extends DataGrid
{
	/** @var string */
	protected $tableClass = '';

	/** @var string */
	protected $sortableGroup;

	protected $groupActionCustomCollection;

	public function __construct(Nette\ComponentModel\IContainer $parent = null, $name = null)
	{
		parent::__construct($parent, $name);
		//		$this->setItemsPerPageList([10, 20, 50, 100, 1000, 'all']);
		$this->setDefaultPerPage(20);
		$this->setRefreshUrl(false);
		$this->setRememberState(true);
		$this->setOuterFilterColumnsCount(4);

		$this->setTemplateFile(__DIR__ . '/templates/datagrid.latte');

		$this->onAnchor[] = function() {
			$this->getTemplate()->newTemplate = __DIR__ . '/templates/datagrid.latte';
		};
	}

	public function render()
	{
		if ($this->isTreeView()) {
			$this->template->extendTemplate = ROOT_DIR . '/vendor/ublaboo/datagrid/src/templates/datagrid_tree.latte';
			$this->setTemplateFile(__DIR__ . '/templates/datagrid_tree.latte');
		} else {
			$this->template->extendTemplate = $this->getOriginalTemplateFile();
		}
		$this->template->tableClass    = $this->tableClass;
		$this->template->sortableGroup = $this->sortableGroup;

		parent::render();
	}

	public function getPerPage()
	{
		$items_per_page_list = $this->getItemsPerPageList();

		$per_page = $this->per_page ?: reset($items_per_page_list);

		if ($per_page !== 'all' && !in_array((int) $this->per_page, $items_per_page_list, true)) {
			$per_page = key($items_per_page_list);
		}

		return $per_page;
	}

	public function addAction($key, $name, $href = null, array $params = null)
	{
		$this->addActionCheck($key);

		$href = $href ?: $key;

		if ($params === null) {
			$params = [$this->primary_key];
		}

		$control = $this->actions[$key] = new Column\Action($this, $href, $name, $params);

		return $control;
	}

	public function createComponentFilter()
	{
		$form = new BaseForm($this, 'filter');

		$form->setMethod(static::$form_method);

		$form->setTranslator($this->getTranslator());

		/**
		 * InlineEdit part
		 */
		$inline_edit_container = $form->addContainer('inline_edit');

		if ($this->inlineEdit instanceof InlineEdit) {
			$inline_edit_container->addSubmit('submit', 'ublaboo_datagrid.save')
				->setValidationScope([$inline_edit_container]);
			$inline_edit_container->addSubmit('cancel', 'ublaboo_datagrid.cancel')
				->setValidationScope(false);

			$this->inlineEdit->onControlAdd($inline_edit_container);
			$this->inlineEdit->onControlAfterAdd($inline_edit_container);
		}

		/**
		 * InlineAdd part
		 */
		$inline_add_container = $form->addContainer('inline_add');

		if ($this->inlineAdd instanceof InlineEdit) {
			$inline_add_container->addSubmit('submit', 'ublaboo_datagrid.save')
				->setValidationScope([$inline_add_container]);
			$inline_add_container->addSubmit('cancel', 'ublaboo_datagrid.cancel')
				->setValidationScope(false)
				->setAttribute('data-datagrid-cancel-inline-add', true);

			$this->inlineAdd->onControlAdd($inline_add_container);
			$this->inlineAdd->onControlAfterAdd($inline_add_container);
		}

		/**
		 * ItemDetail form part
		 */
		$items_detail_form = $this->getItemDetailForm();

		if ($items_detail_form instanceof Nette\Forms\Container) {
			$form['items_detail_form'] = $items_detail_form;
		}

		/**
		 * Filter part
		 */
		$filter_container = $form->addContainer('filter');

		foreach ($this->filters as $filter) {
			$filter->addToFormContainer($filter_container);
		}

		if (!$this->hasAutoSubmit()) {
			$filter_container['submit'] = $this->getFilterSubmitButton();
		}

		/**
		 * Group action part
		 */
		$group_action_container = $form->addContainer('group_action');

		if ($this->hasGroupActions()) {
			$this->getGroupActionCollection()->addToFormContainer($group_action_container);
		}

		$groupActionCustomContainer = $form->addContainer('groupActionCustom');

		if ($this->getGroupActionCustomCollection()) {
			$this->getGroupActionCustomCollection()->addToFormContainer($groupActionCustomContainer);
		}

		if (!$form->isSubmitted()) {
			$this->setFilterContainerDefaults($form['filter'], $this->filter);
		}

		/**
		 * Per page part
		 */
		$form->addSelect('per_page', '', $this->getItemsPerPageList())
			->setTranslator(null);

		if (!$form->isSubmitted()) {
			$form['per_page']->setValue($this->getPerPage());
		}

		$form->addSubmit('per_page_submit', 'ublaboo_datagrid.per_page_submit')
			->setValidationScope([$form['per_page']]);

		$form->onSubmit[] = [$this, 'filterSucceeded'];
	}

	/*******************************************************************************************************************
	 * ==================  Columns
	 */

	/**
	 * @param string $key
	 * @param string $name
	 * @param null   $column
	 *
	 * @return \Ublaboo\DataGrid\Column\Column|Column\ColumnStatus
	 * @throws \Ublaboo\DataGrid\Exception\DataGridException
	 */
	public function addColumnStatus($key, $name, $column = null)
	{
		$this->addColumnCheck($key);
		$column = $column ?: $key;

		return $this->addColumn($key, new Column\ColumnStatus($this, $key, $column, $name));
	}

	/**
	 * @param      $key
	 * @param      $name
	 * @param null $column
	 *
	 * @return \Ublaboo\DataGrid\Column\Column|Column\ColumnPosition
	 */
	public function addColumnPosition($key, $name, $column = null)
	{
		$column = $column ?: $key;

		return $this->addColumn($key, new Column\ColumnPosition($this, $key, $column, $name));
	}

	public function addColumnEmail($key, $name, $column = null)
	{
		$column = $this->addColumnText($key, $name, $column);
		$column->setTemplate(__DIR__ . '/templates/column_email.latte', [
			'key' => $column->getColumn(),
		]);

		return $column;
	}

	public function addColumnPhone($key, $name, $column = null, $country = null, bool $showRegion = false)
	{
		$column = $this->addColumnText($key, $name, $column);

		$column->setAlign('right');
		$column->setTemplate(__DIR__ . '/templates/column_phone.latte', [
			'key'        => $column->getColumn(),
			'keyRegion'  => $country,
			'showRegion' => $showRegion,
		]);

		return $column;
	}

	public function addColumnPrice($key, $name, $column = null)
	{
		$this->addColumnCheck($key);
		$column = $column ?: $key;

		return $this->addColumn($key, new Column\ColumnPrice($this, $key, $column, $name));
	}

	/*******************************************************************************************************************
	 * ==================  Filters
	 */

	public function addFilterRadioNestedList($key, $name, array $flatItems, $column = null)
	{
		$column = $column ?: $column;

		$this->addFilterCheck($key);

		return $this->filters[$key] = new Filter\FilterRadioNestedList($this, $key, $name, $flatItems, $column);
	}

	public function addFilterCheckboxNestedList($key, $name, array $flatItems, $column = null)
	{
		$column = $column ?: $column;

		$this->addFilterCheck($key);

		return $this->filters[$key] = new Filter\FilterCheckboxNestedList($this, $key, $name, $flatItems, $column);
	}

	/*******************************************************************************************************************
	 * ==================  Group actions
	 */

	/**
	 * @param       $title
	 * @param array $options
	 *
	 * @return \Ublaboo\DataGrid\GroupAction\GroupAction
	 */
	public function addGroupSelectCustomAction($title, $options = [])
	{
		return $this->getGroupActionCustomCollection()->addGroupSelectAction($title, $options);
	}

	/*******************************************************************************************************************
	 * ==================  Setters
	 */

	public function setTableClass($class)
	{
		$this->tableClass = $class;

		return $this;
	}

	public function setSortableGroup($group)
	{
		$this->sortableGroup = $group;

		return $this;
	}

	/**
	 * Get collection of all group actions
	 * @return CustomGroupActionCollection
	 */
	public function getGroupActionCustomCollection()
	{
		if (!$this->groupActionCustomCollection) {
			$this->groupActionCustomCollection = new CustomGroupActionCollection($this);
		}

		return $this->groupActionCustomCollection;
	}
}
