<?php declare(strict_types = 1);

namespace Core\AdminModule\Components;

use Core\Model\Helpers\DataToFormInputs;
use Core\Model\Settings;
use Core\Model\SystemConfig;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Nette\Utils\ArrayHash;
use Nette\Utils\Json;

abstract class SettingsForm extends BaseControl
{
	/** @var SystemConfig */
	protected $systemConfig;

	/** @var DataToFormInputs */
	protected $dataToFormInputs;

	/** @var Settings */
	protected $settings;

	/** @var string */
	protected $settingsNamespace;

	/** @var string */
	protected $templateFile;

	public function __construct(string $namespace, SystemConfig $systemConfig, DataToFormInputs $dataToFormInputs, Settings $settings)
	{
		$this->settingsNamespace = $namespace;
		$this->systemConfig      = $systemConfig;
		$this->dataToFormInputs  = $dataToFormInputs;
		$this->settings          = $settings;
		$this->templateFile      = __DIR__ . '/SettingsForm.latte';
	}

	public function render()
	{
		$this->template->thisForm = $this['form'];
		$this->template->setFile($this->templateFile);
	}

	/*******************************************************************************************************************
	 * ======================== COMPONENTS
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addComponent(new BaseContainer(), 'component');
		$form->addSubmit('submit', 'default.save');

		$parsed = $this->dataToFormInputs->parseData($form['component'], $this->getSystemConfigData());

		$form->onSuccess[] = [$this, 'formSuccess'];

		$this->template->componentStructure = $parsed['structure'];

		$defaults    = [];
		$settingsRaw = $this->settings->getAllRaw();
		foreach ($this->getSystemConfigData() as $v) {
			if (!isset($v['name']))
				continue;

			$d = $settingsRaw[$v['name']] ?? null;
			if ($d)
				$defaults[$v['name']] = $d;
		}
		$this->dataToFormInputs->setDefaults($form['component'], $defaults);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		$this->em->beginTransaction();
		try {
			$presenter = $this->getPresenter();

			$vals = [];
			foreach ($form->getComponent('component')->getComponents() as $c) {
				if ($c->getIsMultiLanguage())
					$vals[$c->getName()] = Json::encode($c->getValue());
				else
					$vals[$c->getName()] = $c->getValue();
			}

			$this->settings->saveMulti($vals);

			$presenter->flashMessageSuccess('default.saved');
			$presenter->redrawControl('flashes');

			$this->em->commit();
		} catch (\Exception $e) {
			$this->em->rollback();
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}
		$this->settings->clearCache();
		$this->redrawControl('form');

		return true;
	}

	/**
	 * @return array
	 */
	public function getSystemConfigData(): array
	{
		$result = [];
		foreach ($this->systemConfig->get($this->settingsNamespace, []) as $v) {
			if ($this->settingsNamespace !== 'globalData')
				$v['name'] = $this->settingsNamespace . ucfirst($v['name']);
			$result[] = $v;
		}

		return $result;
	}
}
