<?php declare(strict_types = 1);

namespace Core\Model\TemplateReader;

use Core\Model\Helpers\DataToFormInputs;
use Core\Model\Helpers\Strings;
use Core\Model\TemplateReader\Entity\ITemplate as EntityITemplate;
use Core\Model\Templating\Macros\SilentControl;
use Core\Model\Templating\Macros\TemplateText;
use Core\Model\UI\AbstractPresenter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\Controls\CheckboxListInput;
use Core\Model\UI\Form\Controls\HiddenInput;
use Core\Model\UI\Form\Controls\SelectInput;
use Latte\Engine;
use Nette\Forms\Controls\HiddenField;
use Nette\Forms\IControl;
use Nette\Localization\ITranslator;
use Nette\Utils\FileSystem;
use Nette\Http\Request;
use Nette\Application\UI\ITemplateFactory;
use Core\Model\Images\ImagePipe;
use Pages\Model\Paths;

/**
 * Class TemplateReader
 * @package Core\Model\TemplateReader
 */
class TemplateReader
{
	public $uiControl = null;

	/** @var TemplateTextTypesCollection */
	protected $templateTextTypesCollection;

	/** @var Request */
	protected $httpRequest;

	/** @var ITemplateFactory */
	protected $templateFactory;

	/** @var ImagePipe */
	protected $imagePipe;

	/** @var ITranslator */
	protected $translator;

	/** @var DataToFormInputs */
	protected $dataToFormInputs;

	/** @var Paths */
	protected $pathsService;

	//TODO typ šablony (místo default proměnná)
	protected $templatesDir;

	/** @var string */
	protected $translateKey = 'templateReader';

	/** @var string */
	protected $tmpDir = TMP_DIR . '/cache/latte/templateReader';

	/** @var array */
	protected $componentStructure;

	/**
	 * TemplateReader constructor.
	 *
	 * @param TemplateTextTypesCollection $templateTextTypesCollection
	 * @param Request                     $request
	 * @param ITemplateFactory            $templateFactory
	 * @param ImagePipe                   $imagePipe
	 */
	public function __construct(ITemplateFactory $templateFactory, TemplateTextTypesCollection $templateTextTypesCollection, Request $request, ImagePipe $imagePipe, ITranslator $translator, DataToFormInputs $dataToFormInputs, Paths $paths)
	{
		$this->templateTextTypesCollection = $templateTextTypesCollection;
		$this->httpRequest                 = $request;
		$this->templateFactory             = $templateFactory;
		$this->imagePipe                   = $imagePipe;
		$this->translator                  = $translator;
		$this->dataToFormInputs            = $dataToFormInputs;
		$this->pathsService                = $paths;
		$this->reInitComponentStructure();
	}

	/**
	 * Vrátí strukturu komponent v poli pro vykreslení
	 *
	 * @return array
	 */
	public function getComponentStructure()
	{
		return $this->componentStructure;
	}

	/**
	 * Nastavení klíče pro překlad (admin.key.hodnota)
	 *
	 * @param $key
	 */
	public function setTranslateKey($key)
	{
		$this->translateKey = $key;
	}

	/**
	 * Nastaví výchozí stav struktury pro znovuvytvoření
	 */
	private function reInitComponentStructure()
	{
		$this->componentStructure = ['sections' => []];
	}

	/**
	 * Nastavení složky ze kterých se načtou šablony
	 *
	 * @param $dir
	 */
	public function setTemplatesDir($dir)
	{
		$this->templatesDir = $dir;
	}

	public function getTemplates(?string $keyPrefix = null): array
	{
		$templates = [];
		foreach (glob($this->templatesDir . '/*.latte') as $file) {
			$fileName = pathinfo($file, PATHINFO_FILENAME);
			$tKey     = 'admin.' . $this->translateKey . '.' . $fileName;
			$t        = $this->translator->translate($tKey);

			$k = basename($file);
			if ($keyPrefix)
				$k = $keyPrefix . '|' . $k;

			$templates[$k] = $t == $tKey ? $fileName : $t;
		}

		if (!empty($templates))
			asort($templates, SORT_LOCALE_STRING);

		return $templates;
	}

	/**
	 * Vrátí složku kam se ukládají temp soubory při vytváření struktury
	 *
	 * @return string
	 */
	protected function getTmpDir()
	{
		if ($this->templatesDir) {
			return $this->tmpDir . DS . basename($this->templatesDir);
		}

		return $this->tmpDir;
	}

	/**
	 * Vrártí složku kde jsou šablony
	 *
	 * @param $template
	 *
	 * @return string
	 */
	public function getTemplatePath($template)
	{
		return $this->templatesDir . DS . $template;
	}

	/**
	 * TODO stejné jako v DataToFormInputs. Rozdil jenom v $this->componentStruture
	 *
	 * Přečte nastavenou šablonu a do komponenty formuláře načte potřebné komponenty
	 *
	 * @param BaseContainer $container
	 * @param string        $file
	 */
	public function loadTemplateComponents(BaseContainer &$container, string $file)
	{
		$this->uiControl = $container->lookup(null);
		$this->reInitComponentStructure();

		$result = $this->dataToFormInputs->parseData($container, $this->getTemplateInputs($file));
		if (isset($result['structure']))
			$this->componentStructure = $result['structure'];
	}

	/**
	 * Nastaví výchozí hodnoty formuláři se zvolenou šablonou
	 *
	 * @param BaseContainer         $container
	 * @param EntityITemplate|array $data
	 * @param null                  $file
	 */
	public function setDefaults(BaseContainer $container, $data, string $template, $file = null)
	{
		/** @var IControl[] $components */
		$components       = $container->components;
		$availableLocales = $this->translator->getLocalesWhitelist();

		if ($data instanceof EntityITemplate === false) {
			$tmp = [];

			foreach ($data as $l => $v) {
				if (!in_array($l, $availableLocales)) {
					$tmp[$l] = $v;
					continue;
				}

				foreach ($components as $name => $c) {
					if (property_exists($c, 'isMultiLanguage') && $c->getIsMultiLanguage()) {
						$tmp[$name][$l] = $v->getText($name) ?? null;
					} else {
						$tmp[$name] = $v->getText($name) ?? null;
					}
				}
			}

			$data = $tmp;
		}

		if ($data instanceof EntityITemplate && $file && $template == $file || !$file) {
			foreach ($components as $k => $c) {
				$value = $data instanceof EntityITemplate ? $data->getText($k) : ($data[$k] ?? null);

				if ($c instanceof SelectInput) {
					if (is_array($value)) {
						$c->setDefaultValue($value);
					} else if (array_key_exists($value, $c->getItems())) {
						$c->setDefaultValue($value);
					}
				} else if ($c instanceof CheckboxListInput) {
					$tmp = [];
					foreach ($value as $v) {
						if (array_key_exists($v, $c->getItems()))
							$tmp[] = $v;
					}

					$c->setDefaultValue($tmp);
				} else if ($c instanceof HiddenInput) {
					$c->reSetValue($value);
				} else if ($c instanceof \Nette\Forms\Controls\BaseControl) {
					$c->setDefaultValue($value);
				} else if ($c instanceof BaseContainer) {
					$this->setDefaults($c, $value, $template, $file);
				}
			}
		}
	}

	/**
	 * Zpracuje zvolenou šablonu a vrátí pole s připravenýma hodnotama
	 *
	 * @param $template
	 *
	 * @return array
	 */
	public function getTemplateInputs($template)
	{
		$file        = $this->getTemplatePath($template);
		$fileDir     = dirname($file);
		$content     = @file_get_contents($file);
		$tmpFile     = $this->getTmpDir() . DS . basename($file);
		$saveToCache = [];

		if (!$content)
			return [];

		FileSystem::createDir($this->getTmpDir());

		// Vyhledání všech souborů které se načítají
		$re = '/{(include|import) (.*)}/m';
		preg_match_all($re, $content, $matches, PREG_SET_ORDER, 0);
		foreach ($matches as $match) {
			if (isset($match[2])) {
				$match[2] = explode(',', $match[2])[0];
				if (strpos($match[2], "'") !== 0 && strpos($match[2], "'") !== 0) {
					$match[2] = explode(',', $match[2])[0];
					ob_start();
					eval("echo  $match[2];");
					$path = ob_get_clean();

					$tf      = $this->getTmpDir() . DS . ltrim(str_replace(SRC_DIR, '', $path), DS);
				} else {
					$match[2] = trim($match[2], "'");
					$match[2] = trim($match[2], '"');

					$tf      = $this->getTmpDir() . DS . ltrim($match[2], '../');
					$path = $fileDir . DS . $match[2];
				}
				$tf = addslashes($tf);
				$tf = iconv('ASCII', 'UTF-8//IGNORE', $tf);

				$content = str_replace($match[0], '{include ' . $tf . '}', $content);
				FileSystem::createDir(dirname($tf));
				$saveToCache[$tf] = file_get_contents(dirname($file) . DS . $path);
			}
		}
		$saveToCache[$tmpFile] = $content;

		// Upravení obsahu pro použití v adminu a uložení do temp
		foreach ($saveToCache as $f => $content) {
			$re      = '/{(templateText|tt) (.*)}/m';
			$content = preg_replace($re, "R$0", $content);
			//			preg_match('/toVar="(.*)"/m', $content, $vars);
			//
			//			foreach ($vars as $var)
			//				$content = "{var $var[1] = null}" . $content;

			if (strpos($content, '{block content}') !== false) {
				//				$re = '/\{block content\}(.*?)\{\/block\}/sm';
				//				preg_match($re, $content, $matches, PREG_OFFSET_CAPTURE, 0);

				//				if (isset($matches[1][0]))
				//					$content = $matches[1][0];

			}
			//			$content = str_replace('{block content}', '', $content);
			//			$content = str_replace('{/block}', '', $content);
			$content = str_replace('R{tt', '{ttR', $content);
			$content = str_replace('R{templateText', '{templateTextR', $content);
			$content = preg_replace("/{link(.*)}/", "#", $content);
			$content = preg_replace("/{snippet(.*)}/", "", $content);
			$content = preg_replace("/{\/snippet}/", "", $content);
			$content = preg_replace("/n:snippet=\"(.*)\"/", "", $content);

			// TODO přidat podporu pro komponenty
			$content = preg_replace("/{control(.*)}/", "#", $content);

			file_put_contents($f, $content);
		}

		// Vytvoření čísté šablony pro čtení
		$latte             = $this->templateFactory->createTemplate($this->uiControl);
		$latte->_imagePipe = $this->imagePipe;
		$latte->setFile($tmpFile);
		$content = $latte->renderToString();

		file_put_contents($tmpFile, $content);

		preg_match_all($re, $content, $output, PREG_SET_ORDER, 0);

		$inputs = [];
		foreach ($output as $v) {
			if (!isset($v[2]))
				continue;

			$arr = TemplateText::parseInputAttrs($v[2]);

			if (isset($arr['name']) && isset($arr['title'])) {
				foreach ($arr as $attrKey => $attrValue) {
					// Připravení dat pro překlad
					if (in_array($attrKey, ['title', 'sectionName', 'groupName'])) {
						$tmp = explode(', ', $attrValue, 2);

						if (isset($tmp[1])) {
							$params = [];
							foreach (explode(',', $tmp[1]) as $param) {
								$param = trim($param);

								[$k, $v] = explode('=>', $param);
								$params[$k] = $v;
							}
							$tmp[1] = $params;
						}

						$arr[$attrKey] = $tmp;
					}
				}

				$inputs[] = $arr;
			}
		}

		return $inputs;
	}
}
