<?php declare(strict_types = 1);

namespace Core\FrontModule\Model;

use Contributte\Application\LinkGenerator;
use Core\Model\Providers\ISiteMapImage;
use Core\Model\Providers\ISiteMapUrl;
use Kdyby\Doctrine\EntityManager;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Navigations;
use Nette\Http\Request;
use Thepixeldeveloper\Sitemap as DataSitemap;

/**
 * TODO cache - sledovat všechny entity které mají traitu TSeo a podle toho mazat cache
 *
 * Class SiteMap
 * @package Core\FrontModule\Model
 */
class SiteMap
{
	const CHANGE_FREQ_ALWAYS  = 'always';
	const CHANGE_FREQ_HOURLY  = 'hourly';
	const CHANGE_FREQ_DAILY   = 'daily';
	const CHANGE_FREQ_WEEKLY  = 'weekly';
	const CHANGE_FREQ_MONTHLY = 'monthly';
	const CHANGE_FREQ_YEARLY  = 'yearly';
	const CHANGE_FREQ_NEVER   = 'never';

	/** @var EntityManager */
	protected $em;

	/** @var LinkGenerator */
	protected $linkGenerator;

	/** @var Request */
	protected $httpRequest;

	/** @var Navigations */
	protected $navigations;

	public function __construct(EntityManager $em, LinkGenerator $linkGenerator, Request $request, Navigations $navigations)
	{
		$this->em            = $em;
		$this->linkGenerator = $linkGenerator;
		$this->httpRequest   = $request;
		$this->navigations   = $navigations;
	}

	/**
	 * Prohledá navigaci a její komponenty. Sestaví Xml a vrátí
	 *
	 * @param string $group
	 * @param int    $page
	 *
	 * @return DataSitemap\SitemapIndex|DataSitemap\Urlset|null
	 * @throws \Nette\Application\UI\InvalidLinkException
	 */
	public function getSiteMap($group = 'default', $page = 0)
	{
		/** @var Navigation[] $navigations */
		$navigations = $this->em->getRepository(Navigation::class)->createQueryBuilder('n')
			->andWhere('n.isPublished = 1')->andWhere('n.componentType != \'navigation.customLink\'')
			->andWhere('n.componentType != \'navigation.alias\'')
			->orderBy('n.position', 'ASC')
			->getQuery()->getResult();

		$pageLimit     = 50000;
		$includeImages = false;

		if ($group == 'images') {
			$includeImages = true;
			$group         = 'default';
		}

		// Pár základních proměnných
		$domain = $this->httpRequest->getUrl()->getHostUrl();
		/** @var DataSitemap\Url $urls */
		$urls = [];

		foreach ($navigations as $nav) {
			// Kontrola navigace jestli se může přidat to sitemapy
			if (!$this->canAddToSiteMap($nav))
				continue;

			$params = ['activeNavigation' => $nav];
			if ($group == 'default') {
				$link = $domain . $this->navigations->getUrlById($params);
				if ($includeImages) {
					// TODO obrázky
				} else {
					$urls[$link] = $this->createUrl($link, $nav);
				}
			}

			// Vyhledání komponenty a vytvoření URL pokud obsahuje nějaké položky
			$component = $this->navigations->itemsCollector->getItemById($nav->componentType);
			if ($component) {

				foreach ($component->getComponentChild($nav, true) as $v) {
					if (isset($v['siteMapData'][$group]) && $this->canAddToSiteMap($v['siteMapData'][$group])) {
						$link = $domain . $this->navigations->getUrlById($v);
						$url  = $this->createUrl($link, $v['siteMapData'][$group]);

						if ($includeImages) {
							foreach ($v['siteMapData'][$group]->getImages() as $img) {
								$url->addExtension($this->createImage($domain . $img->getFilePath(), $img));
							}

							if (!empty($v['siteMapData'][$group]->getImages())) {
								$urls[$link] = $url;
							}
						} else {
							$urls[$link] = $url;
						}
					}
				}
			}
		}

		$chunks = array_chunk($urls, $pageLimit);
		if (count($urls) > $pageLimit && $page == 0) {
			$urlSet = new DataSitemap\SitemapIndex();

			foreach ($chunks as $k => $chunk) {
				$chunkLink = $this->linkGenerator->link('Core:Front:Default:sitemap', [
					'group' => $group,
					'page'  => $k + 1,
				]);
				$chunkUrl  = new DataSitemap\Sitemap($chunkLink);
				//TODO last MOD

				$urlSet->add($chunkUrl);
			}
		} else {
			$urlSet = new DataSitemap\Urlset();
			$urls   = $chunks[$page > 0 ? $page - 1 : 0];

			if (!$urls)
				return null;

			foreach ($urls as $v)
				$urlSet->add($v);
		}

		return $urlSet;
	}

	/**
	 * @param string      $url
	 * @param ISiteMapUrl $data
	 *
	 * @return string|DataSitemap\Url
	 */
	protected function createUrl(string $url, $data)
	{
		$url = new DataSitemap\Url($url);
		if ($data->getModified())
			$url->setLastMod($data->getModified());
		if ($data->getSeo('siteMapChangeFreq'))
			$url->setChangeFreq((string) $data->getSeo('siteMapChangeFreq'));
		if ($data->getSeo('siteMapPriority'))
			$url->setPriority((string) $data->getSeo('siteMapPriority'));

		return $url;
	}

	/**
	 * @param string        $url
	 * @param ISiteMapImage $img
	 *
	 * @return DataSitemap\Extensions\Image
	 */
	protected function createImage(string $url, ISiteMapImage $img)
	{
		$image = (new DataSitemap\Extensions\Image($url))
			->setCaption($img->getCaption())
			->setGeoLocation($img->getGeoLocation())
			->setTitle($img->getTitle())
			->setLicense($img->getLicense());

		return $image;
	}

	/**
	 * @param ISiteMapUrl $data
	 *
	 * @return bool
	 */
	protected function canAddToSiteMap($data)
	{
		if (($data instanceof ISiteMapUrl || method_exists($data, 'getSeo')) &&
			($data->getSeo('addToSiteMap') === 0
				|| strpos((string) $data->getSeo('robots'), 'noindex') !== false)
		) {
			return false;
		}

		return true;
	}

	/**
	 * @return array
	 * @throws \ReflectionException
	 */
	public function getAllChangeFreq(): array
	{
		$refl = new \ReflectionClass(self::class);
		$arr  = [];

		foreach ($refl->getConstants() as $k => $v) {
			if (strpos($k, 'CHANGE_FREQ_') === 0)
				$arr[$v] = $v;
		}

		return $arr;
	}
}
