<?php declare(strict_types = 1);

namespace Core\Model\Images;

use Core\Model\MathParser\Math;
use Nette;
use Nette\Utils\Image as NImage;

class ImagePipe
{

	use Nette\SmartObject;

	/** @var string */
	protected $cacheDir;

	/** @var string */
	private $wwwDir;

	/** @var string */
	private $path;

	/** @var array */
	protected $watermark;

	/** @var string */
	private $originalPrefix = "original";

	/** @var string */
	private $baseUrl;

	/** @var string|null */
	protected $namespace = null;

	/** @var array */
	public $onBeforeSaveThumbnail = [];

	/**
	 * @param array              $params
	 * @param Nette\Http\Request $httpRequest
	 */
	public function __construct($params, Nette\Http\Request $httpRequest)
	{
		$this->wwwDir    = $params['wwwDir'] ?? WWW_DIR;
		$this->cacheDir  = $params['cacheDir'] ?? WWW_DIR . '/thumbs';
		$this->watermark = $params['watermark'] ?? null;
		$this->baseUrl   = rtrim($httpRequest->url->baseUrl, '/');
	}

	/**
	 * @param $path
	 */
	public function setPath($path)
	{
		$this->path = $path;
	}

	/**
	 * @param $dir
	 */
	public function setCacheDir($dir)
	{
		$this->cacheDir = $dir;
	}

	/**
	 * @return string
	 */
	public function getCacheDir()
	{
		return $this->cacheDir;
	}

	/**
	 * @return string
	 */
	public function getPath()
	{
		return $this->path !== null ? $this->path : $this->baseUrl . str_replace($this->wwwDir, '', $this->cacheDir);
	}

	/**
	 * @param $originalPrefix
	 */
	public function setOriginalPrefix($originalPrefix)
	{
		$this->originalPrefix = $originalPrefix;
	}

	/**
	 * @return string
	 */
	public function getOriginalPrefix()
	{
		return $this->originalPrefix;
	}

	/**
	 * @throws \Nette\InvalidStateException
	 */
	private function checkSettings()
	{
		if ($this->cacheDir == null) {
			throw new Nette\InvalidStateException("Assets directory is not setted");
		}
		if (!file_exists($this->cacheDir)) {
			throw new Nette\InvalidStateException("Assets directory '{$this->cacheDir}' does not exists");
		} elseif (!is_writeable($this->cacheDir)) {
			throw new Nette\InvalidStateException("Make assets directory '{$this->cacheDir}' writeable");
		}
		if ($this->getPath() == null) {
			throw new Nette\InvalidStateException("Path is not setted");
		}
	}

	/**
	 * @param $namespace
	 *
	 * @return $this
	 */
	public function setNamespace($namespace)
	{
		if (empty($namespace)) {
			$this->namespace = null;
		} else {
			$this->namespace = $namespace . "/";
		}

		return $this;
	}

	/**
	 * @param string $image
	 * @param null   $size
	 * @param null   $flags
	 * @param bool   $watermark
	 * @param bool   $strictMode
	 *
	 * @return string
	 * @throws Nette\Utils\UnknownImageFileException
	 * @throws \Latte\CompileException
	 */
	public function request($image, $size = null, $flags = null, $watermark = false, $strictMode = false)
	{
		$remoteUrl = null;
		if (strpos($image, 'http') === 0) {
			$remoteUrl = new Nette\Http\Url($image);
			$image     = $remoteUrl->getDomain() . $remoteUrl->getPath();
		}

		$image = ltrim($image, '/');
		$this->checkSettings();
		if ($image instanceof ImageProvider) {
			$this->setNamespace($image->getNamespace());
			$image = $image->getFilename();
		} elseif (empty($image)) {
			return "#";
		}

		if ($size === null) {
			return $image;
		}

		[$width, $height] = explode("x", $size);
		$cropX = '50%';
		$cropY = '50%';

		if ($flags == null) {
			$flags = NImage::FIT;
		} elseif (strpos($flags, 'crop') === 0) {
			$crop = ltrim($flags, 'crop');
			[$cropX, $cropY] = explode('x', $crop);
			$cropX = abs($cropX);
			$cropY = abs($cropY);
			$flags = 'crop';
		} elseif (!is_int($flags)) {
			switch (strtolower($flags)) {
				case "fit":
					$flags = NImage::FIT;
					break;
				case "fill":
					$flags = NImage::FILL;
					break;
				case "exact":
					$flags = NImage::EXACT;
					break;
				case "shrink_only":
					$flags = NImage::SHRINK_ONLY;
					break;
				case "stretch":
					$flags = NImage::STRETCH;
					break;
			}

			if (!isset($flags)) {
				throw new \Latte\CompileException('Mode is not allowed');
			}
		}

		$f = "{$flags}_{$width}x{$height}" . ($watermark ? '_wm' : '');
		if ($cropX != '50%' && $cropY != '50%')
			$f .= '_' . ((int) $cropX) . 'x' . ((int) $cropY);
		$thumbFile    = $this->cacheDir . "/" . dirname($image) . "/{$f}/" . basename($image);
		$thumbPath    = str_replace($this->cacheDir, '', $thumbFile);
		$originalFile = $remoteUrl ? $remoteUrl->getAbsoluteUrl() : $this->wwwDir . "/" . $image;

		if (!file_exists($thumbFile)) {
			$this->mkdir(dirname($thumbFile));
			if (file_exists($originalFile) || $remoteUrl) {
				$img = NImage::fromFile($originalFile);

				$exif = exif_read_data($originalFile);
				if (isset($exif['Orientation'])) {
					switch ($exif['Orientation']) {
						case 8:
							$img->rotate(90, 0);
							break;
						case 3:
							$img->rotate(180, 0);
							break;
						case 6:
							$img->rotate(-90, 0);
							break;
					}
				}
				
				if ($flags === "crop") {
					$aspectRatio = $width / $height;
					$imgWidth    = $img->getWidth();
					$imgHeight   = $img->getHeight();

					if ($imgWidth > $imgHeight) {
						$cropWidth  = round(($imgHeight / 100) * ($width / $height) * 100);
						$cropHeight = $imgHeight;
					} else if ($imgWidth < $imgHeight) {
						$cropWidth  = $imgWidth;
						$cropHeight = round(($imgWidth / 100) * ($height / $width) * 100);
					} else {
						$cropWidth  = $imgWidth;
						$cropHeight = $imgHeight;
					}

					$img->crop($cropX, $cropY, $cropWidth, $cropHeight)->resize($width, $height, NImage::EXACT);
					$img->sharpen();
				} else {
					$img->resize($width, $height, $flags | NImage::SHRINK_ONLY);
				}

				if ($watermark)
					$this->addWatermark($img);

				$this->onBeforeSaveThumbnail($img, $this->namespace, $image, $width, $height, $flags);
				$img->save($thumbFile);
			} elseif ($strictMode) {
				throw new FileNotFoundException;
			}
		}
		$this->namespace = null;

		return $this->getPath() . $thumbPath;
	}

	/**
	 * @param string $dir
	 *
	 * @throws \Nette\IOException
	 * @return void
	 */
	private static function mkdir($dir)
	{
		$oldMask = umask(0);
		@mkdir($dir, 0777, true);
		@chmod($dir, 0777);
		umask($oldMask);

		if (!is_dir($dir) || !is_writable($dir)) {
			throw new Nette\IOException("Please create writable directory $dir.");
		}
	}

	/**
	 * @param $file
	 */
	public function removeThumbs($file)
	{
		$dir = str_replace($this->wwwDir, '', $file);
		$dir = pathinfo($dir, PATHINFO_DIRNAME);

		foreach (glob($this->cacheDir . $dir . '/**/' . basename($file)) as $file) {
			@unlink($file);
		}
	}

	protected function addWatermark(NImage &$img)
	{
		if (!isset($this->watermark['file'], $this->watermark['size'], $this->watermark['left'], $this->watermark['top']) || !file_exists($this->watermark['file']))
			return null;

		$watermark = NImage::fromFile($this->watermark['file']);
		$math      = new Math();

		$size  = str_replace('$imgWidth', $img->getWidth(), $this->watermark['size']);
		$width = $math->evaluate($size);

		$left = str_replace(['$imgWidth', '$wmWidth', ','], [$img->getWidth(), $width, '.'], $this->watermark['left']);
		$left = $math->evaluate($left);
		$top  = str_replace(['$imgWidth', '$wmWidth', ','], [$img->getWidth(), $width, '.'], $this->watermark['top']);
		$top  = $math->evaluate($top);

		$watermark->resize($width, 0, NImage::FIT);
		$img->place($watermark, $left, $top, $this->watermark['opacity'] ?? 80);
	}
}
