<?php declare(strict_types = 1);

namespace Core\Model\UI;

use Contributte\Application\UI\BasePresenter;
use Core\Components\Flashes\Flashes;
use Core\Components\Flashes\IFlashesFactory;
use Core\Components\Header\IFaviconFactory;
use Core\Components\Header\IHeaderFactory;
use Core\Components\Header\IMetaFactory;
use Core\Components\Header\ITitleFactory;
use Core\Model\Images\ImagePipe;
use Core\Model\Lang\Lang;
use Core\Model\Lang\Langs;
use Kdyby\Autowired\AutowireComponentFactories;
use Kdyby\Translation\Translator;

abstract class AbstractPresenter extends BasePresenter
{
	use AutowireComponentFactories;
	use TUITemplate;

	/** @persistent */
	public $locale;

	/** @var Translator @inject */
	public $translator;

	/** @var ImagePipe @inject */
	public $imagePipe;

	/** @var Langs @inject */
	public $langsService;

	/** @var Lang */
	protected $lang;

	/** @var string */
	protected $baseUrl;

	protected function startup()
	{
		parent::startup();

		$localeAvailable = false;
		foreach ($this->translator->getAvailableLocales() as $l) {
			if (strpos($l, $this->locale) === 0) {
				$localeAvailable = true;
				break;
			}
		}

		if (!$localeAvailable) {
			$this->locale = $this->translator->getDefaultLocale();
		}

		foreach ($this->translator->getAvailableLocales() as $l) {
			if (strpos($l, $this->locale) === 0) {
				SetLocale(LC_ALL, $l . ".utf8");
				break;
			}
		}

		$this->baseUrl = $this->getHttpRequest()->getUrl()->getBaseUrl();
		$this->lang    = $this->langsService->getLang($this->translator->getLocale());
	}

	public function beforeRender()
	{
		parent::beforeRender();
		$this->template->title = $this['title']->getTitle();
	}

	/*******************************************************************************************************************
	 *=========================== Setters
	 */
	protected function setTitle($title, $colonText = null)
	{
		$this['title']->setTitle($title, false);
		$this['title']->setColonText($colonText);

		//TODO remove
		if (isset($this['header']) && empty($this['header']->getTitle()))
			$this->setHeader($title);
	}

	protected function setHeader($title = null, $icon = null, $description = null)
	{
		if (!empty($title)) {
			$this['header']->setTitle($this->translator->translate($title));
			if (!$this['title']->getTitle())
				$this['title']->setTitle($this->translator->translate($title));
		} else $this['header']->setTitle($this['title']->getTitle());
		if (!empty($icon))
			$this['header']->setIcon($icon); else $this['header']->setIcon('fas fa-table');
		if (!empty($description))
			$this['header']->setDescription($this->translator->translate($description));
	}

	/*******************************************************************************************************************
	 *=========================== Components
	 */

	protected function createComponentTitle(ITitleFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentFlashes(IFlashesFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentMeta(IMetaFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentFavicon(IFaviconFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentHeader(IHeaderFactory $factory)
	{
		return $factory->create();
	}

	/*******************************************************************************************************************
	 *=========================== Helpers
	 */

	public function getFullModuleName()
	{
		$arr = explode(':', $this->name);
		array_pop($arr);

		return implode(':', $arr);
	}

	public function t($message, $count = null, $parameters = [], $domain = null, $locale = null)
	{
		return $this->translator->translate($message, $count, $parameters, $domain, $locale);
	}

	/*******************************************************************************************************************
	 *=========================== Flashes
	 */

	public function flashMessageSuccess($message)
	{
		return parent::flashMessage($message, Flashes::FLASH_SUCCESS);
	}

	public function flashMessageInfo($message)
	{
		return parent::flashMessage($message, Flashes::FLASH_INFO);
	}

	public function flashMessageWarning($message)
	{
		return parent::flashMessage($message, Flashes::FLASH_WARNING);
	}

	public function flashMessageDanger($message)
	{
		return parent::flashMessage($message, Flashes::FLASH_DANGER);
	}

	/*******************************************************************************************************************
	 *=========================== GET
	 */

	/**
	 * @return Lang
	 */
	public function getLang()
	{
		return $this->lang;
	}

	/**
	 * @return Langs
	 */
	public function getLangsService()
	{
		return $this->langsService;
	}
}
