<?php declare(strict_types = 1);

namespace Core\AdminModule\Components\Redirects;

use Core\AdminModule\Model\Redirects;
use Core\Model\Entities\Redirect;
use Core\Model\Helpers\FormHelper;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Application\ForbiddenRequestException;
use Nette\Utils\ArrayHash;

class RedirectForm extends BaseControl
{
	/** @var int|null @persistent */
	public ?int $redirectId = null;

	protected Redirects $redirectsService;

	public function __construct(Redirects $redirectsService)
	{
		$this->redirectsService = $redirectsService;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('title', 'core.redirects.title');
		$form->addText('from', 'core.redirects.from')
			->setRequired();
		$form->addText('to', 'core.redirects.to')
			->setRequired();
		$form->addText('package', 'core.redirects.package');
		$form->addText('relationKey', 'core.redirects.relationKey');
		$form->addText('relationValue', 'core.redirects.relationValue');
		$form->addSelect('redirectCode', 'core.redirects.redirectCode', [
			301 => 301,
			302 => 302,
		]);

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$values->from = ltrim($values->from, '/');
			$values->to   = ltrim($values->to, '/');
			$redirect     = $this->redirectId ? $this->redirectsService->get($this->redirectId) : new Redirect($values->title, $values->from, $values->to);

			FormHelper::fillEntityByValues($redirect, $values);
			$redirect->isPublished = 1;
			$redirect->isLocked    = 0;

			$this->em->persist($redirect);
			$this->em->flush();

			$form->addCustomData('redirectId', $redirect->getId());

			$this->getPresenter()->flashMessageSuccess($this->t('default.saved'));

			return true;
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			$this->redrawControl('form');
		}

		return false;
	}

	public function setRedirect(int $id)
	{
		$redirect = $this->redirectsService->get($id);

		if (!$redirect)
			throw new ForbiddenRequestException("Redirect '$id' not found");

		$this->redirectId = $id;

		FormHelper::fillContainerByEntity($this['form'], $redirect);
	}
}
