<?php declare(strict_types = 1);

namespace Core\AdminModule\Components\Redirects;

use Core\AdminModule\Model\Redirects;
use Core\Model\Entities\Redirect;
use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Http\FileUpload;
use Nette\Utils\ArrayHash;
use PhpOffice\PhpSpreadsheet\IOFactory;

class UploadForm extends BaseControl
{
	protected Redirects $redirectsService;

	public array $onSuccessUpload = [];

	public function __construct(Redirects $redirectsService)
	{
		$this->redirectsService = $redirectsService;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();
		$form->setShowLangSwitcher(false);

		$form->addUpload('file', 'core.redirects.file')
			->setRequired();

		$form->addText('txtSeparator', 'core.redirects.txtSeparator');
		$form->addText('xlsTabName', 'core.redirects.xlsTabName');
		$form->addText('xlsTitleCol', 'core.redirects.xlsTitleCol');
		$form->addText('xlsFromCol', 'core.redirects.xlsFromCol');
		$form->addText('xlsToCol', 'core.redirects.xlsToCol');
		$form->addText('xlsSkipRows', 'core.redirects.xlsSkipRows');

		$form->addSubmit('upload', 'default.upload');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			/** @var FileUpload $file */
			$file = $values->file;

			if (!$file->isOk())
				throw new \Exception($this->t('core.redirects.fileError'));

			$extension = pathinfo($file->getName(), PATHINFO_EXTENSION);
			$data      = [];

			if ($extension === 'txt' && $values->txtSeparator) {
				$fh = fopen($file->getTemporaryFile(), 'r');
				while ($line = fgets($fh)) {
					$tmp = explode($values->txtSeparator, $line);
					if (count($tmp) < 3)
						continue;

					$data[] = new Redirect(trim($tmp[0]), trim($tmp[1]), trim($tmp[2]));
				}
				fclose($fh);
			} else if (in_array($extension, ['xls', 'xlsx']) && $values->xlsFromCol && $values->xlsToCol) {
				$spreadsheet = IOFactory::load($file->getTemporaryFile());
				$spreadsheet->setActiveSheetIndexByName($values->xlsTabName);

				$sheet    = $spreadsheet->getActiveSheet();
				$lastRow  = $sheet->getHighestRow();
				$skipRows = array_map(fn($v) => intval($v), explode(',', (string) $values->xlsSkipRows));

				for ($r = 1; $r <= $lastRow; $r++) {
					if (in_array($r, $skipRows))
						continue;

					$title = $values->xlsTitleCol ? $sheet->getCellByColumnAndRow((int) $values->xlsTitleCol, $r)->getValue() : '';
					$from  = $sheet->getCellByColumnAndRow((int) $values->xlsFromCol, $r)->getValue();
					$to    = $sheet->getCellByColumnAndRow((int) $values->xlsToCol, $r)->getValue();

					if (!$from || !$to)
						continue;

					foreach (['from', 'to'] as $v) {
						if (Strings::startsWith(${$v}, 'http')) {
							$tmp = explode('//', ${$v});

							if (isset($tmp[1])) {
								$tmp = explode('/', $tmp[1], 2);

								if (isset($tmp[1]))
									${$v} = trim($tmp[1], '/');
							}
						}
					}

					$data[] = new Redirect(trim($title), trim($from), trim($to));
				}
			}

			if (!empty($data)) {
				foreach (array_chunk($data, 100) as $chunk) {
					foreach ($chunk as $entity) {
						/** @var Redirect $entity */
						$entity->isLocked    = 0;
						$entity->isPublished = 1;
						$this->em->persist($entity);
					}
					$this->em->flush();
				}
			}

			@unlink($file->getTemporaryFile());

			$this->getPresenter()->flashMessageSuccess('default.uploaded');
			$this->getPresenter()->redrawControl('flashes');

			$form->reset();
			$this->redrawControl('form');

			foreach ($this->onSuccessUpload as $c) {
				$c();
			}

			return true;
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			$this->redrawControl('form');
		}

		return false;
	}
}
