<?php declare(strict_types = 1);

namespace Core\AdminModule\Components\Settings;

use Core\AdminModule\Model\Sites;
use Core\Model\Entities\SiteText;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;

class SiteForm extends BaseControl
{
	protected string $ident;

	protected Sites $sitesService;

	public function __construct(string $ident, Sites $sites)
	{
		$this->ident        = $ident;
		$this->sitesService = $sites;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function attached($presenter): void
	{
		parent::attached($presenter);

		$site = $this->sitesService->get($this->ident);
		if ($site) {
			$d = [
				'logo'              => $site->logo,
				'siteNameSeparator' => $site->siteNameSeparator,
				'email'             => $site->email,
			];

			foreach ($site->texts->toArray() as $l => $v) {
				$d['domain'][$l]   = $v->domain;
				$d['siteName'][$l] = $v->siteName;
				$d['isActive'][$l] = $v->isActive;
			}

			$this['form']->setDefaults($d);
		}
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax()
			->setShowLangSwitcher(false);

		$form->addBool('isActive', 'admin.site.isActive')
			->setIsMultilanguage();
		$form->addText('domain', 'admin.site.domain')
			->setMaxLength(255)
			->setIsMultilanguage();
		$form->addFilesManager('logo', 'admin.site.logo');
		$form->addText('siteName', 'admin.site.siteName')
			->setMaxLength(255)
			->setIsMultilanguage();
		$form->addText('siteNameSeparator', 'admin.site.siteNameSeparator')
			->setNullable()
			->setDefaultValue('|')
			->setMaxLength(255);
		$form->addEmail('email', 'admin.site.email')
			->setMaxLength(255)
			->setRequired();

		$form->addSubmit('submit', 'default.save');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			$site       = $this->sitesService->get($this->ident);
			/** @var SiteText[] $texts */
			$texts = $site->texts->toArray();

			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l]))
					$texts[$l] = new SiteText($site, $v['domain'], $l);

				$texts[$l]->domain   = $v['domain'];
				$texts[$l]->siteName = $v['siteName'];
				$texts[$l]->isActive = (int) $v['isActive'];

				$this->em->persist($texts[$l]);
			}

			$site->setTexts($texts);
			$site->logo              = $values->logo;
			$site->siteNameSeparator = $values->siteNameSeparator;
			$site->email             = $values->email;

			$this->em->persist($site);
			$this->em->flush();

			$this->em->getConfiguration()->getResultCacheImpl()->delete('siteTextCache');

			$this->getPresenter()->flashMessageSuccess('admin.site.dataUpdated');
			$this->getPresenter()->redrawControl('flashes');

			$cache = new Cache($this->cacheStorage, \Core\Model\Sites::CACHE_NAMESPACE);
			$cache->remove(\Core\Model\Sites::CACHE_NAMESPACE);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}
}
