<?php declare(strict_types = 1);

namespace Core\FrontModule\Components;

use Core\Model\Parameters;
use Core\Model\Sites;
use Core\Model\SystemConfig;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\Model\Mailing\TemplateFactory;
use Nette\Mail\IMailer;
use Nette\Mail\Message;
use Nette\Mail\SmtpMailer;
use Nette\Utils\ArrayHash;
use Nette\Utils\Validators;

/**
 * Class ContactForm
 * @package Core\FrontModule\Components
 */
class ContactForm extends BaseControl
{
	/** @var IMailer */
	protected $mailer;

	/** @var TemplateFactory */
	protected $templateFactory;

	/** @var Message */
	protected $message;

	/** @var array */
	protected $data;

	/** @var Sites */
	protected Sites $sites;

	public function __construct($data, IMailer $mailer, TemplateFactory $templateFactory, Sites $sites)
	{
		$this->message = new Message();
		if (Parameters::load('mailSender.enableSmtp', false) && Parameters::load('mailSender.smtp', null))
			$this->mailer = new SmtpMailer(Parameters::load('mailSender.smtp'));
		else
			$this->mailer = $mailer;
		$this->templateFactory = $templateFactory;
		$this->data            = $data;
		$this->sites           = $sites;

		$this->setSubject($data['subject'] ?? '');

		if (is_array($data['replyTo'])) {
			foreach ($data['replyTo'] as $email => $name) {
				if (Validators::isEmail($email))
					$this->addReplyTo($email, $name);
			}
		}

		if (isset($data['bcc']) && Validators::isEmail($data['bcc'])) {
			foreach (explode(',', $data['bcc']) as $v) {
				$this->message->addBcc(trim($v));
			}
		}
	}

	public function render(): void
	{
		$this->template->thisForm = $this['form'];
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Setters
	 */
	public function setFrom($fromEmail, $fromName = null)
	{
		$this->message->setFrom($fromEmail, $fromName);

		return $this;
	}

	public function addTo($toEmail, $toName = null)
	{
		$this->message->addTo($toEmail, $toName);

		return $this;
	}

	public function setSubject($subject)
	{
		$this->message->setSubject($subject);

		return $this;
	}

	public function addReplyTo($email, $name = null)
	{
		$this->message->addReplyTo($email, $name);

		return $this;
	}

	public function signalReceived($signal): void
	{
		parent::signalReceived($signal);
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();
		$form->setAjax();
		$form->setAction($this->getPresenter()->getHttpRequest()->getUrl()->getAbsoluteUrl());

		$form->addText('name', 'default.contactForm.name')->setPlaceholder('default.contactForm.name')->setRequired();
		$form->addEmail('email', 'default.contactForm.email')->setPlaceholder('default.contactForm.email')->setRequired();
		$form->addText('phone', 'default.contactForm.phone')->setPlaceholder('default.contactForm.phone');
		$form->addTextArea('message', 'default.contactForm.message')
			->setHtmlAttribute('placeholder', $this->t('default.contactForm.message'))
			->setRequired();
		if (Parameters::load('contactForm.addGdprAgree'))
			$form->addCheckbox('gdpr', 'GDPR')->setRequired();

		if ($this->configService->get('useGoogleRecaptcha') === true) {
			if (SystemConfig::load('googleRecaptchaInvisible') === true)
				$form->addInvisibleReCaptcha('recaptcha', true, 'default.formMessages.recaptcha');
			else
				$form->addReCaptcha('recaptcha', 'Google recaptcha', true, 'default.formMessages.recaptcha')
					->setRequired('default.formMessages.recaptchaRequired');
		}

		$form->addSubmit('submit', 'default.contactForm.send');

		$form->onValidate[] = [$this, 'formOnValidate'];
		$form->onSuccess[]  = [$this, 'formOnSuccess'];

		return $form;
	}

	public function formOnValidate(BaseForm $form, ArrayHash $values)
	{
		if ($this->configService->get('useGoogleRecaptcha') === true && $form['recaptcha']->hasErrors()) {
			$form->addError('default.formMessages.recaptcha');
		}
		if ($form->getErrors()) {
			$this->redrawControl('formErrors');
		}
	}

	public function formOnSuccess($form, ArrayHash $values)
	{
		try {
			$template     = $this->templateFactory->create($this->sites->getCurrentSite()->getIdent());
			$templateFile = TEMPLATES_DIR . '/Front/default/ContactFormEmail.latte';
			$template->setFile(file_exists($templateFile) ? $templateFile : __DIR__ . '/ContactFormEmail.latte');

			$currentSite = $this->sites->getCurrentSite();
			$data        = $this->data;

			$toEmail = $data['toEmail'] == 'currentSite' ? $currentSite->getEmail() : $data['toEmail'];
			$toName  = $data['toName'] == 'currentSite' ? $currentSite->getSiteName() : $data['toName'];

			$fromEmail = $data['fromEmail'] == 'currentSite' ? $toEmail : $data['fromEmail'];
			$fromName  = $data['fromName'] == 'currentSite' ? $toName : $data['fromName'];

			$this->setFrom($fromEmail ?? null, $fromName ?? null);
			$this->addTo($toEmail ?? null, $toName ?? null);

			if (is_string($this->data['replyTo']) && $this->data['replyTo'] == 'formEmail'
				|| is_array($this->data['replyTo']) && in_array('formEmail', $this->data['replyTo']))
				$this->addReplyTo($values->email, $values->name);

			$params            = (array) $values;
			$params['subject'] = $this->message->getSubject();
			$template->setParameters((array) $values);

			$this->message->setHtmlBody($template->renderToString());
			$this->mailer->send($this->message);

			if (isset($this->data['sendCopyToSender']) && $this->data['sendCopyToSender'] == true) {
				try {
					$this->message->clearHeader('To');
					$this->message->clearHeader('Reply-To');
					$this->message->clearHeader('Bcc');
					$this->message->addTo($values->email, $values->name);
					$this->message->addReplyTo($fromEmail, $fromName);
					$this->message->setSubject($this->translator->translate('default.contactForm.senderSubject') . ' ' . $currentSite->getSiteName());

					$templateFile = TEMPLATES_DIR . '/Front/default/ContactFormEmail_Sender.latte';
					$template->setFile(file_exists($templateFile) ? $templateFile : __DIR__ . '/ContactFormEmail_Sender.latte');

					$this->message->setHtmlBody($template->renderToString());
					$this->mailer->send($this->message);
				} catch (\Exception $e) {
				}
			}

			$this->template->okMessage = $this->t('default.contactForm.formOk');
		} catch (\Exception $e) {
			$form->addError($this->t('default.contactForm.sendingError'));

			return false;
		}

		if ($this->getPresenter()->isAjax()) {
			$this->redrawControl('form');
		} else {
			$this->getPresenter()->redirectUrl($this->getPresenter()->getHttpRequest()->getUrl()->getAbsoluteUrl());
		}
	}
}
