<?php declare(strict_types = 1);

namespace Core\Model\Helpers;

use Core\Model\SystemConfig;
use libphonenumber\PhoneNumberFormat;
use libphonenumber\PhoneNumberUtil;
use \Nette\Utils\Strings as NetteStrings;

class Strings extends NetteStrings
{
	public static bool $priceKeepTail = false;

	/**
	 * @param float       $price
	 * @param string|null $currency
	 * @param string|null $separator
	 * @param bool        $currencyFromLeft
	 * @param string|null $decSep
	 * @param int|null    $decimals
	 *
	 * @return string
	 */
	public static function priceFormat(float $price, ?string $currency = 'Kč', ?string $separator = ' ', bool $currencyFromLeft = false, ?string $decSep = ',', ?int $decimals = 2): string
	{
		$decPoint       = $decSep;
		$priceFormatted = number_format($price, $decimals, $decPoint, ' ');

		// e.g.: ,00
		$tail = $decPoint . implode('', array_fill(0, $decimals, 0));

		if (!self::$priceKeepTail && NetteStrings::endsWith($priceFormatted, $tail)) {
			$priceFormatted = substr($priceFormatted, 0, strlen($priceFormatted) - strlen($tail));
		}

		return sprintf(
			'%s%s%s',
			$currencyFromLeft ? $currency : $priceFormatted,
			$separator ?? ' ',
			$currencyFromLeft ? $priceFormatted : $currency
		);
	}

	/**
	 * @param string      $value
	 * @param string|null $valueRegion
	 * @param bool        $showRegion
	 *
	 * @return string
	 */
	public static function phoneFormat(string $value, ?string $valueRegion = null, bool $showRegion = false): string
	{
		try {
			$phoneUtils  = PhoneNumberUtil::getInstance();
			$valid       = false;
			$phone       = null;
			$region      = null;
			$plusInValue = false;

			if ($valueRegion)
				$regions = [$valueRegion] + $phoneUtils->getSupportedRegions();
			else
				$regions = SystemConfig::load('phoneInputRegions') + $phoneUtils->getSupportedRegions();

			foreach ($regions as $region) {
				if (substr($value, 0, 1) === '+') {
					$phone       = $phoneUtils->parse($value);
					$plusInValue = true;
				} else {
					$phone = $phoneUtils->parse($value, strtoupper($region));
				}

				$valid = $phoneUtils->isValidNumber($phone);

				if ($valid) {
					$region = $phoneUtils->getRegionCodeForCountryCode($phone->getCountryCode());
					break;
				} else if ($plusInValue)
					break;
			}

			return $valid && $phone
				? $phoneUtils->format($phone, PhoneNumberFormat::INTERNATIONAL) . ($showRegion ? ' (' . $region . ')' : '')
				: $value;
		} catch (\Exception $e) {
		}

		return $value;
	}

	public static function joinPaths(...$paths): string
	{
		return preg_replace('~[/\\\\]+~', DIRECTORY_SEPARATOR, implode(DIRECTORY_SEPARATOR, $paths));
	}

	public static function isValidTimestamp($timestamp): bool
	{
		return ((string) (int) $timestamp === $timestamp)
			&& ($timestamp <= PHP_INT_MAX)
			&& ($timestamp >= ~PHP_INT_MAX);
	}

	public static function formatEntityDecimal($value): string
	{
		return str_replace(',', '.', (string) $value);
	}

	public static function normalizeText(string $value): string
	{
		$value = strip_tags($value);
		$value = str_replace('&nbsp;', ' ', $value);
		$value = preg_replace('/\s\s+/', ' ', $value);

		return $value;
	}

	public static function wordsCount(string $value): int
	{
		return count(explode(' ', $value));
	}
}
