<?php declare(strict_types = 1);

namespace Core\Model;

use Core\Model\Entities\SiteText;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Router\RouteHelper;
use Navigations\Model\NavigationConfig;
use Nette\Application\BadRequestException;
use Nette\Http\Request;
use Nette\Http\Response;
use Core\Model\Dao;
use Nette\Http\Url;
use Nette\Localization\ITranslator;

class Sites
{
	/** @var EntityManagerDecorator */
	protected $em;

	/** @var Request */
	protected $httpRequest;

	/** @var Response */
	protected $httpResponse;

	/** @var ITranslator */
	protected $translator;

	/** @var Dao\Site[] */
	protected $cSites;

	protected ?Dao\Site $cCurrent = null;

	public static $currentIdentOverride = null;

	public static $currentLangOverride = null;

	public function __construct(EntityManagerDecorator $em, Request $request, Response $response, ITranslator $translator)
	{
		$this->em           = $em;
		$this->httpRequest  = $request;
		$this->httpResponse = $response;
		$this->translator   = $translator;
	}

	/**
	 * @return Dao\Site|null
	 */
	public function getCurrentSite(): ?Dao\Site
	{
		$current = $this->cCurrent;

		if (self::$currentIdentOverride)
			$current = $this->getSites()[self::$currentIdentOverride];

		if (!$current) {
			$host          = $this->httpRequest->getUrl()->getHost();
			$foundSite     = null;
			$alternateSite = null;

			foreach ($this->getSites() as $site) {
				$domain = $site->getDomains()[$this->translator->getLocale()] ?? null;

				if ($domain && $host == $domain->getDomain()) {
					$site->currentLang = $domain->getLang();
					$foundSite         = $site;
					break;
				}

				if (!$alternateSite)
					foreach ($site->getDomains() as $domain) {
						if ($host == $domain->getDomain()) {
							$site->currentLang = $domain->getLang();
							$alternateSite     = $site;
						}
					}
			}

			if (!$alternateSite)
				$alternateSite = array_values($this->getSites())[0];

			$this->cCurrent = $foundSite ?: $alternateSite;
		} else {
			$this->cCurrent = $current;
		}

		if (self::$currentLangOverride && isset($this->cCurrent->getDomains()[self::$currentLangOverride]))
			$this->cCurrent->currentLang = self::$currentLangOverride;

		return $this->cCurrent;
	}

	public function validateAndRedirectSite(): void
	{
		if (php_sapi_name() === 'cli' || !$this->getSites())
			return;

		$host      = $this->httpRequest->getUrl()->getHost();
		$hostParts = explode('.', $host, 2);
		$pathParts = explode('/', ltrim($this->httpRequest->getUrl()->getPath(), '/'));
		$urlLang   = strlen($pathParts[0]) === 2 ? $pathParts[0] : null;

		if (NavigationConfig::load('useCZinUrl') && $urlLang === 'cz')
			$urlLang = 'cs';

		foreach ($this->getSites() as $site) {
			foreach ($site->getDomains() as $lang => $domain) {
				$hostFound = false;

				if ($host === $domain->domain) {
					$hostFound = 'prod';
				}

				if ($hostFound !== false) {
					if ($urlLang && $urlLang == $domain->getLang() && $domain->isDefault && !NavigationConfig::load('showDefaultLangInUrl')) {
						header('Location: https://' . $domain->domain);
						exit;
					} else if (!$urlLang) {
						$this->setNavigationLang($domain->getLang());
						$site->currentLang = $domain->getLang();
						$this->translator->setLocale($domain->getLang());
					}

					if ($urlLang) {
						if ($urlLang != $domain->getLang()) {
							$tmp       = $site->getDomains()[$urlLang] ?? null;
							$tmpDomain = $tmp ? $tmp->domain : null;
							if ($tmp && $tmpDomain && $tmpDomain !== $host) {
								header('Location: https://' . $tmpDomain);
								exit;
							}

							$this->setNavigationLang($urlLang);
							$site->currentLang = $urlLang;
							$this->translator->setLocale($urlLang);
						}
					}

					return;
				}

				$parts = explode('.', $domain->getDomain(), 2);

				// host je bez www ale doména s
				if (isset($parts[1]) && $host === $parts[1])
					$this->redirectToHost($domain->domain);

				// host je s www ale doména bez
				if ($hostParts[1] === $domain->getDomain())
					$this->redirectToHost($domain->domain);
			}
		}

		throw new BadRequestException('domainNotFound');
	}

	protected function setNavigationLang(string $lang): void
	{
		if ($lang === 'cz')
			$lang = 'cs';
		RouteHelper::$validatedLang = $lang;
	}

	/**
	 * @return Dao\Site[]
	 */
	public function getSites(bool $onlyActive = true): array
	{
		$cKey = $onlyActive ? 'active' : 'all';
		if ($this->cSites[$cKey] === null) {
			$this->cSites[$cKey] = [];

			$qb = $this->em->getRepository(SiteText::class)->createQueryBuilder('st')
				->addSelect('s.ident, st.domain, st.siteName, s.siteNameSeparator, s.logo, s.email, st.lang, st.isDefault')
				->innerJoin('st.site', 's')
				->orderBy('st.isDefault', 'DESC');

			if ($onlyActive)
				$qb->andWhere('st.isActive = 1');

			foreach ($qb->getQuery()
				         ->enableResultCache(3600, 'siteTextCache')
				         ->getScalarResult() as $row) {

				if (!isset($this->cSites[$cKey][$row['ident']])) {
					$site        = new Dao\Site($row['ident']);
					$site->logo  = $row['logo'] ?: null;
					$site->email = $row['email'];

					$this->cSites[$cKey][$row['ident']] = $site;
				} else {
					$site = $this->cSites[$cKey][$row['ident']];
				}

				$domain                    = new Dao\SiteDomain($site, $row['domain'], $row['lang'], (int) $row['isDefault']);
				$domain->siteName          = $row['siteName'];
				$domain->siteNameSeparator = $row['siteNameSeparator'];
			}
		}

		return $this->cSites[$cKey];
	}

	protected function redirectToHost(string $host, $path = true): void
	{
		$url    = $this->httpRequest->getUrl();
		$newUrl = new Url($url);
		$newUrl->setHost($host);
		if (is_string($path))
			$newUrl->setPath($path);
		elseif ($path === null)
			$newUrl->setPath('');

		$this->httpResponse->redirect($newUrl->getAbsoluteUrl());
		exit;
	}
}
