<?php declare(strict_types = 1);

namespace Core\Model\UI\Form;

use Core\Model\Lang\Langs;
use Core\Model\UI\Form\Traits\BaseContainerTrait;
use Core\Model\UI\Form\Traits\BootstrapContainerTrait;
use Nette\Application\UI\Form;
use Nette\ComponentModel\IContainer;
use Nette\Forms\IFormRenderer;
use Nette\InvalidArgumentException;
use Nette\Utils\Html;
use Nextras;

/**
 * Class BaseForm
 * @package Core\Model\UI\Form
 *
 */
class BaseForm extends Form
{
	use BootstrapContainerTrait;
	use BaseContainerTrait;

	/** @var string|Html */
	public $description;

	/** @var callable[] */
	public $onSuccessSave;

	/** @var callable[] */
	public $onSuccessSaveAndClose;

	/** @var callable[] */
	public $onCancel;

	/** @var bool */
	protected $disableSuccessSave = false;

	/** @var bool */
	protected $isAjax = true;

	/** @var array */
	protected $customData = [];

	/** @var Langs */
	public $langsService;

	/** @var bool */
	protected $showLangSwitcher = true;

	protected array $headLinks = [];

	/**
	 * @param int|IContainer|null $container
	 */
	public function __construct($container = null, $name = null)
	{
		parent::__construct($container, $name);
		$this->setRenderer(new BootstrapRenderer);
	}

	protected function beforeRender()
	{
		parent::beforeRender();
		foreach ($this->getControls() as $control) {
			if ($control->isRequired()) {
				$control->getLabelPrototype()->class('required', true);
			}
		}
	}

	/**
	 * @return int
	 */
	public function getRenderMode()
	{
		return $this->getRenderer()->getMode();
	}

	/**
	 * @return BootstrapRenderer|\Nette\Forms\IFormRenderer
	 */
	public function getRenderer(): IFormRenderer
	{
		$renderer = parent::getRenderer();
		$renderer->setForm($this);

		return $renderer;
	}

	/**
	 * @param IFormRenderer $renderer
	 *
	 * @return void
	 */
	public function setRenderer(IFormRenderer $renderer = null): void
	{
		if (!$renderer instanceof BootstrapRenderer) {
			throw new InvalidArgumentException('Must be a BootstrapRenderer');
		}
		parent::setRenderer($renderer);
	}

	/**
	 * @return bool if form is ajax. True by default.
	 */
	public function isAjax()
	{
		return $this->isAjax;
	}

	/**
	 * @param bool $isAjax
	 *
	 * @return BaseForm
	 */
	public function setAjax($isAjax = true)
	{
		$this->isAjax = $isAjax;

		if ($isAjax)
			$this->getForm()->getElementPrototype()->class[] = 'ajax';

		return $this;
	}

	/**
	 * @param int $renderMode
	 */
	public function setRenderMode($renderMode)
	{
		$this->getRenderer()->setMode($renderMode);
	}

	/**
	 * @param bool $disable
	 */
	public function disableSuccessSave(bool $disable = true) { $this->disableSuccessSave = $disable; }

	public function fireEvents(): void
	{
		parent::fireEvents();

		if ($this->isValid() && !$this->disableSuccessSave) {
			$submittedBy = $this->isSubmitted()->name;

			if ($submittedBy == 'save')
				$this->onSuccessSave($this);
			else if ($submittedBy == 'saveAndClose')
				$this->onSuccessSaveAndClose($this);
		}
	}

	/**
	 * @param array $arr
	 *
	 * @return $this
	 */
	public function setCustomData($arr)
	{
		$this->customData = $arr;

		return $this;
	}

	/**
	 * @param string $key
	 * @param mixed  $value
	 *
	 * @return $this
	 */
	public function addCustomData($key, $value)
	{
		$this->customData[$key] = $value;

		return $this;
	}

	/**
	 * @param string $key
	 *
	 * @return mixed
	 */
	public function getCustomData($key) { return $this->customData[$key] ?: null; }

	/**
	 * @param $service
	 *
	 * @return $this
	 */
	public function setLangsService($service)
	{
		$this->langsService = $service;

		return $this;
	}

	/**
	 * @return Html|null
	 */
	public function getDescription(): ?Html
	{
		if (!$this->description)
			return null;

		return $this->description instanceof Html ? $this->description : Html::el()->setHtml($this->description);
	}

	/**
	 * @param string|Html $description
	 *
	 * @return $this
	 */
	public function setDescription($description): self
	{
		$this->description = $description;

		return $this;
	}

	/**
	 * @param array $inputs
	 * @param array $values
	 *
	 * @return array
	 */
	public function prepareMultilangTextsForEntity($inputs, $values)
	{
		$mTexts = [];
		$langs = array_keys($this->langsService->getLangs(false));
		foreach ($inputs as $k => $v) {
			if (is_numeric($k))
				$k = $v;

			foreach ($values->$k as $lang => $val) {
				if (in_array($lang, $langs)) {
					$mTexts[$lang][$v] = $val;
				}
			}
		}

		return $mTexts;
	}

	/**
	 * @param array  $inputs
	 * @param object $texts
	 *
	 * @return array
	 */
	public function prepareMultilangTextsForForm($inputs, $texts)
	{
		$mTexts  = [];
		$seoData = [];

		foreach ($texts as $lang => $text) {
			foreach ($inputs as $k => $v) {
				if (is_numeric($k))
					$k = $v;

				$mTexts[$k][$lang] = $text->$v;
			}

			if (property_exists(get_class($text), 'seo'))
				foreach ($text->getSeo() as $k => $v)
					$seoData[$k][$lang] = $v;
		}

		return ['texts' => $mTexts, 'seo' => $seoData];
	}

	public function convertMultilangValuesToArray(): array
	{
		$result = [];
		$values = $this->getValues(true);

		foreach ($this->getComponents(true) as $control) {
			$name = $control->getName();

			if (in_array($name, ['saveControl']) || $control instanceof BaseContainer)
				continue;

			if (isset($control->isMultiLanguage) && $control->isMultiLanguage == true) {
				$parents = [];
				$con     = $control;
				while ($con->getParent() instanceof BaseContainer) {
					$con       = $con->getParent();
					$parents[] = $con->getName();
				}

				$parents = array_reverse($parents);
				$val     = $values;

				foreach ($parents as $parent) {
					$val = $val[$parent];
				}

				foreach ($val[$name] as $l => $v) {
					if (!isset($result[$l]))
						$result[$l] = [];

					$target = &$result[$l];
					foreach ($parents as $parent) {
						if (!isset($target[$parent])) {
							$target[$parent] = [];
						}

						$target = &$target[$parent];
					}

					$target[$name] = $v;
				}
			}
		}

		return $result;
	}

	public function getShowLangSwitcher(): bool { return $this->showLangSwitcher; }

	public function setShowLangSwitcher(bool $show = true): self
	{
		$this->showLangSwitcher = $show;

		return $this;
	}

	public function getHeadLinks(): array { return $this->headLinks; }

	public function addHeadLink(string $link, string $text, ?string $class = null): self
	{
		$this->headLinks[] = [
			'link'  => $link,
			'text'  => $text,
			'class' => $class,
		];

		return $this;
	}
}
