<?php declare(strict_types = 1);

namespace Core\FrontModule\Model;

use Contributte\Application\LinkGenerator;
use Core\Model\Providers\ISiteMapImage;
use Core\Model\Providers\ISiteMapUrl;
use Kdyby\Doctrine\EntityManager;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Navigations;
use Navigations\Model\Providers\INavigationItem;
use Nette\Http\Request;
use Nette\Utils\FileSystem;
use Thepixeldeveloper\Sitemap as DataSitemap;

/**
 * TODO cache - sledovat všechny entity které mají traitu TSeo a podle toho mazat cache
 *
 * Class SiteMap
 * @package Core\FrontModule\Model
 */
class SiteMap
{
	const CHANGE_FREQ_ALWAYS  = 'always';
	const CHANGE_FREQ_HOURLY  = 'hourly';
	const CHANGE_FREQ_DAILY   = 'daily';
	const CHANGE_FREQ_WEEKLY  = 'weekly';
	const CHANGE_FREQ_MONTHLY = 'monthly';
	const CHANGE_FREQ_YEARLY  = 'yearly';
	const CHANGE_FREQ_NEVER   = 'never';

	const TMP_DIR = TMP_DIR . '/sitemap';

	/** @var EntityManager */
	protected $em;

	/** @var LinkGenerator */
	protected $linkGenerator;

	/** @var Request */
	protected $httpRequest;

	/** @var Navigations */
	protected $navigations;

	/** @var array */
	protected $componentChildC = [];

	public function __construct(EntityManager $em, LinkGenerator $linkGenerator, Request $request, Navigations $navigations)
	{
		$this->em            = $em;
		$this->linkGenerator = $linkGenerator;
		$this->httpRequest   = $request;
		$this->navigations   = $navigations;
	}

	/**
	 * Prohledá navigaci a její komponenty. Sestaví Xml a vrátí
	 *
	 * @param string $group
	 * @param int    $page
	 *
	 * @return DataSitemap\SitemapIndex|DataSitemap\Urlset|null
	 * @throws \Nette\Application\UI\InvalidLinkException
	 */
	public function getSiteMap($group = 'default', $page = 0)
	{
		$oGroup = $group;
		/** @var Navigation[] $navigations */
		$navigations = $this->em->getRepository(Navigation::class)->createQueryBuilder('n')
			->andWhere('n.isPublished = 1')->andWhere('n.componentType != \'navigation.customLink\'')
			->andWhere('n.componentType != \'navigation.alias\'')
			->orderBy('n.position', 'ASC')
			->getQuery()->getResult();

		$pageLimit = 50000;

		// Pár základních proměnných
		$domain = $this->httpRequest->getUrl()->getHostUrl();
		/** @var DataSitemap\Url $urls */
		$urls      = [];
		$urlsChunk = [];
		$totalUrl  = 0;
		$ci        = 1;
		$files     = [];
		FileSystem::createDir(self::TMP_DIR . '/' . $oGroup . '/');
		$toFile = false;

		foreach (glob(self::TMP_DIR . '/' . $oGroup . '/*') as $k => $f)
			$files[$k + 1] = $f;

		if (empty($files)) {
			foreach ($navigations as $nav) {
				// Kontrola navigace jestli se může přidat to sitemapy
				if (!$this->canAddToSiteMap($nav))
					continue;

				$params = ['activeNavigation' => $nav];
				$link   = $domain . $this->navigations->getUrlById($params);
				// TODO obrázky
				$urls[$link] = $this->createUrl($link, $nav);

				// Vyhledání komponenty a vytvoření URL pokud obsahuje nějaké položky
				$component = $this->navigations->itemsCollector->getItemById($nav->componentType);
				if ($component) {
					if (property_exists($component, 'linkGenerator'))
						$component->linkGenerator = $this->linkGenerator;

					foreach ($this->getComponentChild($component, $nav, true) as $v) {
						if (isset($v['activeNavigation']))
							unset($v['activeNavigation']);

						if (isset($v['siteMapData'][$group]) && $this->canAddToSiteMap($v['siteMapData'][$group])) {
							$link = $domain . $this->navigations->getUrlById($v);
							$url  = $this->createUrl($link, $v['siteMapData'][$group]);

							foreach ($v['siteMapData'][$group]->getImages() as $img) {
								$url->addExtension($this->createImage($domain . $img->getFilePath(), $img));
							}
							$urls[$link] = $url;
							$totalUrl++;

							if (count($urls) > $pageLimit) {
								if ($toFile) {
									$file       = self::TMP_DIR . "/$oGroup/m_$ci.txt";
									$files[$ci] = $file;
									file_put_contents($file, serialize($urls));
								} else {
									$urlsChunk[$ci] = $urls;
								}
								$urls = [];
								$ci++;
							}
						}
					}
				}
			}

			if (!empty($urls)) {
				if ($toFile) {
					$file       = self::TMP_DIR . "/$oGroup/m_$ci.txt";
					$files[$ci] = $file;
					file_put_contents($file, serialize($urls));
				} else {
					$urlsChunk[$ci] = $urls;
				}
				$urls = [];
			}

			if ($toFile)
				file_put_contents(self::TMP_DIR . "/$oGroup/count.txt", $totalUrl);
			else
				$totalUrl = $totalUrl;
		} else {
			$totalUrl = (int) file_get_contents(self::TMP_DIR . "/$oGroup/count.txt");
		}

		if ($totalUrl > $pageLimit && $page == 0) {
			$urlSet = new DataSitemap\SitemapIndex();

			if ($toFile) {
				foreach ($files as $fp => $file) {
					$chunkLink = $this->linkGenerator->link('Core:Front:Default:sitemap', [
						'group' => $oGroup,
						'page'  => $fp,
					]);
					$chunkUrl  = new DataSitemap\Sitemap($chunkLink);
					//TODO last MOD

					$urlSet->add($chunkUrl);
				}
			} else {
				foreach ($urlsChunk as $fp => $url) {
					$chunkLink = $this->linkGenerator->link('Core:Front:Default:sitemap', [
						'group' => $oGroup,
						'page'  => $fp,
					]);
					$chunkUrl  = new DataSitemap\Sitemap($chunkLink);
					//TODO last MOD

					$urlSet->add($chunkUrl);
				}
			}
		} else {
			$urlSet = new DataSitemap\Urlset();
			if ($toFile) {
				$content = file_get_contents(self::TMP_DIR . "/$oGroup/m_" . ($page > 0 ? $page : 1) . ".txt");

				$urls = $content ? unserialize($content) : null;
			} else {
				$urls = $urlsChunk[$page > 0 ? $page : 1];
			}
			$content = null;

			if (!$urls)
				return null;

			foreach ($urls as $v)
				$urlSet->add($v);
		}

		return $urlSet;
	}

	/**
	 * @param string      $url
	 * @param ISiteMapUrl $data
	 *
	 * @return string|DataSitemap\Url
	 */
	protected function createUrl(string $url, $data)
	{
		$url = new DataSitemap\Url($url);
		if ($data->getModified())
			$url->setLastMod($data->getModified());
		if ($data->getSeo('siteMapChangeFreq'))
			$url->setChangeFreq((string) $data->getSeo('siteMapChangeFreq'));
		if ($data->getSeo('siteMapPriority'))
			$url->setPriority((string) $data->getSeo('siteMapPriority'));

		return $url;
	}

	/**
	 * @param string        $url
	 * @param ISiteMapImage $img
	 *
	 * @return DataSitemap\Extensions\Image
	 */
	protected function createImage(string $url, ISiteMapImage $img)
	{
		$image = (new DataSitemap\Extensions\Image($url))
			->setCaption($img->getCaption())
			->setGeoLocation($img->getGeoLocation())
			->setTitle($img->getTitle())
			->setLicense($img->getLicense());

		return $image;
	}

	/**
	 * @param ISiteMapUrl $data
	 *
	 * @return bool
	 */
	protected function canAddToSiteMap($data)
	{
		if (($data instanceof ISiteMapUrl || method_exists($data, 'getSeo')) &&
			($data->getSeo('addToSiteMap') === 0
				|| strpos((string) $data->getSeo('robots'), 'noindex') !== false)
		) {
			return false;
		}

		return true;
	}

	/**
	 * @return array
	 * @throws \ReflectionException
	 */
	public function getAllChangeFreq(): array
	{
		$refl = new \ReflectionClass(self::class);
		$arr  = [];

		foreach ($refl->getConstants() as $k => $v) {
			if (strpos($k, 'CHANGE_FREQ_') === 0)
				$arr[$v] = $v;
		}

		return $arr;
	}

	/**
	 * @param INavigationItem $component
	 * @param Navigation      $nav
	 * @param bool            $sitemapData
	 *
	 * @return mixed
	 */
	public function getComponentChild(INavigationItem $component, $nav, $sitemapData = true)
	{
		if (!isset($this->componentChildC[$nav->getId()]))
			$this->componentChildC[$nav->getId()] = $component->getComponentChild($nav, $sitemapData);

		return $this->componentChildC[$nav->getId()];
	}
}
