<?php declare(strict_types = 1);

namespace Core\FrontModule\Components;

use Core\Model\Mailing\MailBuilderFactory;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Latte\Engine;
use Nette\Application\UI\ITemplateFactory;
use Nette\Mail\IMailer;
use Nette\Mail\Message;
use Nette\Mail\SendmailMailer;
use Nette\Utils\ArrayHash;
use Nette\Utils\Validators;

/**
 * Class ContactForm
 * @package Core\FrontModule\Components
 */
class ContactForm extends BaseControl
{
	/** @var IMailer */
	protected $mailer;

	/** @var ITemplateFactory */
	protected $templateFactory;

	/** @var Message */
	protected $message;

	/** @var array */
	protected $data;

	public function __construct($data, IMailer $mailer, ITemplateFactory $templateFactory, MailBuilderFactory $mailBuilderFactory)
	{
		$this->message         = new Message();
		$this->mailer          = $mailBuilderFactory->create();
		$this->templateFactory = $templateFactory;
		$this->data            = $data;

		$this->setFrom($data['fromEmail'] ?? null, $data['fromName'] ?? null);
		$this->addTo($data['toEmail'] ?? null, $data['toName'] ?? null);
		$this->setSubject($data['subject'] ?? '');

		if (is_array($data['replyTo'])) {
			foreach ($data['replyTo'] as $email => $name) {
				if (Validators::isEmail($email))
					$this->addReplyTo($email, $name);
			}
		}

		if (isset($data['bcc'])) {
			foreach (explode(',', $data['bcc']) as $v) {
				$this->message->addBcc(trim($v));
			}
		}
	}

	public function render()
	{
		$this->template->thisForm = $this['form'];
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Setters
	 */
	public function setFrom($fromEmail, $fromName = null)
	{
		$this->message->setFrom($fromEmail, $fromName);

		return $this;
	}

	public function addTo($toEmail, $toName = null)
	{
		$this->message->addTo($toEmail, $toName);

		return $this;
	}

	public function setSubject($subject)
	{
		$this->message->setSubject($subject);

		return $this;
	}

	public function addReplyTo($email, $name = null)
	{
		$this->message->addReplyTo($email, $name);

		return $this;
	}

	public function signalReceived($signal)
	{
		parent::signalReceived($signal);
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('name', 'default.contactForm.name')->setPlaceholder('default.contactForm.name')->setRequired();
		$form->addEmail('email', 'default.contactForm.email')->setPlaceholder('default.contactForm.email')->setRequired();
		$form->addText('phone', 'default.contactForm.phone')->setPlaceholder('default.contactForm.phone');
		$form->addTextArea('message', 'default.contactForm.message')->setAttribute('placeholder', $this->t('default.contactForm.message'))->setRequired();
		$form->addCheckbox('gdpr', 'GDPR')->setRequired();

		if ($this->configService->get('useGoogleRecaptcha') === true) {
			$form->addInvisibleReCaptcha('recaptcha')->setRequired('default.formMessages.recaptcha');
		}

		$form->addSubmit('submit', 'default.contactForm.send');

		$form->onValidate[] = [$this, 'formOnValidate'];
		$form->onSuccess[]  = [$this, 'formOnSuccess'];

		return $form;
	}

	public function formOnValidate(BaseForm $form, ArrayHash $values)
	{
		if ($form->getErrors()) {
			$this->redrawControl('formErrors');
		}
	}

	public function formOnSuccess($form, ArrayHash $values)
	{
		try {
			$template     = $this->templateFactory->createTemplate();
			$templateFile = TEMPLATES_DIR . '/Front/default/ContactFormEmail.latte';

			if (is_string($this->data['replyTo']) && $this->data['replyTo'] == 'formEmail'
				|| is_array($this->data['replyTo']) && in_array('formEmail', $this->data['replyTo']))
				$this->addReplyTo($values->email, $values->name);

			$this->mailer->setTemplateFile(file_exists($templateFile) ? $templateFile : __DIR__ . '/ContactFormEmail.latte');
			$this->mailer->setMessage($this->message);
			$this->mailer->getTemplate()->setParameters((array) $values);
			$this->mailer->getTemplate()->setParameters(['subject' => $this->message->getSubject()]);

			$this->mailer->send($this->message);
			$this->template->okMessage = $this->t('default.contactForm.formOk');
		} catch (\Exception $e) {
			$form->addError($this->t('default.contactForm.sendingError'));

			return false;
		}

		if ($this->getPresenter()->isAjax()) {
			$this->redrawControl('form');
		} else {
			$this->redirect('this');
		}
	}
}
