<?php declare(strict_types = 1);

namespace Core\AdminModule\Presenters;

use Core\AdminModule\Components\Navigation\INavigationFactory as INavigationSidebarFactory;
use Core\AdminModule\Components\Navigation\Navigation as NavigationSidebar;
use Core\AdminModule\Model\Sites;
use Core\Components\FilesManager\FilesManager;
use Core\Components\FilesManager\IFilesManagerFactory;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Components\Navigation\INavigationFactory;
use Core\Components\Navigation\Navigation;
use Core\Model\Config;
use Core\Model\Module;
use Core\Model\Parameters;
use Core\Model\SystemConfig;
use Core\Model\UI\TUITemplate;
use Core\Model\Entities\EntityManagerDecorator;

/**
 * Class BasePresenter
 * @package Core\AdminModule\Presenters
 */
abstract class BasePresenter extends CustomBasePresenter
{
	use TUITemplate;

	/** @var EntityManagerDecorator @inject */
	public $em;

	/** @var Sites @inject */
	public $sitesService;

	public ?DaoNavigationItem $activeNavigation = null;

	protected function startup()
	{
		parent::startup();
		$this->getHttpResponse()->setHeader('Content-Security-Policy', '');

		$this['navigationSidebar']->setActiveNav();
		if ($this->activeNavigation && $this->activeNavigation->securityConfig === false)
			$this->error('Access forbidden');
	}

	public function checkRequirements($element): void
	{
		parent::checkRequirements($element);
		if (!$this->getUser()->isLoggedIn() && $this->getName() != 'Users:Admin:Login')
			$this->forward(':Users:Admin:Login:');
		if ($this->getUser()->isLoggedIn() && !$this->getUser()->getIdentity()->isActive) {
			$this->getUser()->logout(true);
			$this->redirect(':Users:Admin:Login:');
		}
	}

	protected function performanceModeEnabled(): bool { return false; }

	/*******************************************************************************************************************
	 * =============================== Render
	 */

	public function beforeRender()
	{
		parent::beforeRender();

		$this['meta']->setMeta('robots', 'noindex, nofollow');

		$js = [];
		foreach (glob(ASSETS_DIR . '/admin2/js-autoload/*.js') as $file) {
			$js[] = [
				'file' => str_replace(WWW_DIR, '', $file),
				'v'    => filemtime($file),
			];
		}
		$this->template->autoloadJs = $js;

		$js = [];
		foreach (glob(ASSETS_DIR . '/admin2/js-autoload/head/*.js') as $file) {
			$js[] = [
				'file' => str_replace(WWW_DIR, '', $file),
				'v'    => filemtime($file),
			];
		}
		$this->template->autoloadHeadJs = $js;

		if (file_exists(TEMPLATES_DIR . '/Admin/default/favicon.latte'))
			$this->template->faviconFile = TEMPLATES_DIR . '/Admin/default/favicon.latte';

		$this->template->sites                  = $this->sitesService->getAll();
		$this->template->performanceModeEnabled = $this->performanceModeEnabled() ? 'true' : 'false';
		$this->template->ckEditorSettings       = [
			'forcePasteAsPlainText' => SystemConfig::load('ckeditor.settings.forcePasteAsPlainText', false),
		];
	}

	/*******************************************************************************************************************
	 * =============================== Handle
	 */

	public function handleLogout()
	{
		$defaultRoute = Parameters::load('defaultAdminRoute');
		$this->getUser()->logout(true);
		$this->redirect(":{$defaultRoute['module']}:{$defaultRoute['presenter']}:");
	}

	public function handleSetPerformanceMode(): void
	{
		$this->session->getSection('core')->performanceMode = (bool) $this->getHttpRequest()->getPost('val');
	}

	/*******************************************************************************************************************
	 * =============================== Components
	 */

	protected function createComponentNavigationSidebar(INavigationSidebarFactory $factory): NavigationSidebar
	{
		return $factory->create();
	}

	protected function createComponentNavigation(INavigationFactory $factory): Navigation
	{
		$control = $factory->create();

		return $control;
	}

	protected function createComponentFilesManager(IFilesManagerFactory $factory): FilesManager
	{
		return $factory->create();
	}

	/*******************************************************************************************************************
	 * =============================== Get / Set
	 */
}
