<?php declare(strict_types = 1);

namespace Core\DI;

use Contributte\DI\Helper\ExtensionDefinitionsHelper;
use Contributte\Scheduler\CallbackJob;
use InvalidArgumentException;
use Nette;
use Nette\DI\Compiler;
use Nette\DI\Definitions\Definition;
use Nette\DI\Definitions\Statement;
use Nette\DI\Helpers as DIHelpers;
use Nette\Utils\FileSystem;
use Nette\Utils\Validators;
use Nettrine\ORM\DI\Helpers\MappingHelper;
use Nettrine\ORM\DI\Traits\TEntityMapping;
use Tracy\Debugger;

abstract class CompilerExtension extends Nette\DI\CompilerExtension
{
	protected static int $schedulerJobKey = 0;

	protected $extensionName;

	protected ?string $cDIDir = null;

	public function setCompiler(Compiler $compiler, string $name)
	{
		parent::setCompiler($compiler, $name); // TODO: Change the autogenerated stub

		FileSystem::createDir(TMP_DIR);
		FileSystem::createDir(LOG_DIR);
		FileSystem::createDir(UPLOADS_DIR);
		FileSystem::createDir(THUMBS_DIR);

		$extensionsFile = $this->getDIDir() . '/extensions.neon';
		if (file_exists($extensionsFile)) {
			$extensions = Nette\Neon\Neon::decode(file_get_contents($extensionsFile));

			foreach ($extensions ?? [] as $name => $class) {
				if (is_int($name))
					$name = null;

				$args = [];
				if ($class instanceof Nette\DI\Definitions\Statement)
					[$class, $args] = [$class->getEntity(), $class->arguments];

				if (!is_a($class, \Nette\DI\CompilerExtension::class, true))
					throw new Nette\DI\InvalidConfigurationException("Extension '$class' not found or is not Nette\\DI\\CompilerExtension descendant.");

				$this->compiler->addExtension($name, (new \ReflectionClass($class))->newInstanceArgs($args));
			}
		}

		return $this;
	}

	public function loadConfiguration()
	{
		parent::loadConfiguration();

		$builder = $this->getContainerBuilder();
		$driver  = $builder->getDefinition('orm.annotations.annotationDriver');
		$driver->addSetup('addPaths', [$this->getEntityMappings()]);
	}

	public function setConfig(array $config): self
	{
		$this->config = $config;
		$builder      = $this->getContainerBuilder();

		if (isset($this->getConfig()['parameters'])) {
			$builder->parameters = array_replace_recursive($this->getConfig()['parameters'], $builder->parameters);
		}

		return $this;
	}

	public function beforeCompile()
	{
		parent::beforeCompile();
		$builder = $this->getContainerBuilder();
		$config  = $this->getConfig();

		$translateResources = $this->getTranslationResources();
		if ($translateResources) {
			/** @var Nette\DI\Definitions\ServiceDefinition $translator */
			$translator = $builder->getDefinition('translation.translator');

			$neonDoctrineLoader = $builder->getDefinition('core.staticTexts');

			if (!Debugger::$productionMode)
				$tracyPanel = $builder->getDefinition('translation.tracyPanel');
			foreach ($translateResources as $k1 => $v1) {
				if (!file_exists($v1))
					continue;
				foreach (Nette\Utils\Finder::find('*.neon')->from($v1) as $v2) {
					$match = Nette\Utils\Strings::match($v2->getFilename(), '~^(?P<domain>.*?)\.(?P<locale>[^\.]+)\.(?P<format>[^\.]+)$~');

					if (!$match) {
						continue;
					}

					$translator->addSetup('addResource', [$match['format'], $v2->getPathname(), $match['locale'],
						$match['domain']]);

					$neonDoctrineLoader->addSetup('addResource', [$match['locale'], $match['domain'],
						$v2->getPathname()]);

					if (isset($tracyPanel)) {
						$tracyPanel->addSetup('addResource', [$match['format'], $v2->getPathname(), $match['locale'],
							$match['domain']]);
					}
				}
			}
		}

		// Decorator
		if (isset($config['decorator'])) {
			foreach ($config['decorator'] as $type => $info) {
				$setups = $info['setup'];
				foreach ($this->findByType($type) as $def) {
					foreach ($setups as $setup) {
						foreach ($setup->arguments as $argK => $argV)
							$setup->arguments[$argK] = DIHelpers::expand($argV, $builder->parameters, true);

						if (is_array($setup))
							$setup = new Nette\DI\Statement(key($setup), array_values($setup));

						$def->addSetup($setup);
					}

					if (isset($info['inject'])) {
						Validators::assertField($info, 'inject', 'bool');
						$def->addTag(Nette\DI\Extensions\InjectExtension::TAG_INJECT, $info['inject']);
					}
				}
			}
		}

		$extName = $this->getExtensionName();
		if ($extName !== 'Core') {
			$dir = SRC_DIR . DS . strtolower($extName) . '/src/Model/Entities';

			if (file_exists($dir))
				MappingHelper::of($this)
					->addAnnotation($extName . '\Model\Entities', $dir);
		}

		if (isset($config['scheduler']['jobs'])) {
			$definitionHelper    = new ExtensionDefinitionsHelper($this->compiler);
			$schedulerDefinition = $builder->getDefinition('scheduler.scheduler');

			foreach ($config['scheduler']['jobs'] as $jobName => $jobConfig) {
				if (is_numeric($jobName)) {
					$jobName = self::$schedulerJobKey;
					self::$schedulerJobKey++;
				}

				if (is_array($jobConfig) && (isset($jobConfig['cron']) || isset($jobConfig['callback']))) {
					if (!isset($jobConfig['cron'], $jobConfig['callback'])) {
						throw new InvalidArgumentException(sprintf('Both options "callback" and "cron" of %s > jobs > %s must be configured', $this->name, $jobName));
					}

					$jobDefinition = new Statement(CallbackJob::class, [$jobConfig['cron'], $jobConfig['callback']]);
				} else {
					$jobPrefix     = $this->prefix('job.' . $jobName);
					$jobDefinition = $definitionHelper->getDefinitionFromConfig($jobConfig, $jobPrefix);
					if ($jobDefinition instanceof Definition) {
						$jobDefinition->setAutowired(false);
					}
				}

				$schedulerDefinition->addSetup('add', [$jobDefinition, $jobName]);
			}
		}
	}

	public function afterCompile(Nette\PhpGenerator\ClassType $class)
	{
		parent::afterCompile($class);
		$init = $class->methods["initialize"];

		$ref = new \ReflectionClass(static::class);

		$dir = dirname($ref->getFileName()) . '/../';
		$init->addBody(sprintf('define("PACKAGE_%s_DIR", \'%s\');', strtoupper($this->getExtensionName()), $dir));
	}

	protected function setMapping($mapping)
	{
		$builder = $this->getContainerBuilder();

		//				$arr = [];
		//				foreach ($mapping as $k => $v) {
		//					$override = preg_replace("/(.*)Presenter$/", "$1PresenterOverride", $v);
		//
		//					if ($override != $v)
		//						$arr[$k][] = $override;
		//
		//					$arr[$k][] = $v;
		//				}

		$builder->getDefinition('nette.presenterFactory')->addSetup('setMapping', [$mapping]);
	}

	public function getTranslationResources()
	{
		return [$this->getDIDir() . DS . '..' . DS . 'lang'];
	}

	public function getEntityMappings()
	{
		$arr = [];

		$path = $this->getDIDir() . DS . '..' . DS . 'Model' . DS . 'Entities';
		if ($this->getExtensionName() && file_exists($path))
			$arr[$this->getExtensionName() . "\Model\Entities"] = $path;

		return $arr;
	}

	private function getExtensionName()
	{
		if (!$this->extensionName)
			$this->extensionName = explode('\\', get_called_class())[0];

		return $this->extensionName;
	}

	private function findByType($type)
	{
		return array_filter($this->getContainerBuilder()->getDefinitions(), function($def) use ($type) {
			return is_a($def->getType(), $type, true) || is_a($def->getImplement(), $type, true);
		});
	}

	protected function getDIDir()
	{
		if ($this->cDIDir === null) {
			$obj          = new \ReflectionClass($this);
			$this->cDIDir = dirname($obj->getFileName());
		}

		return $this->cDIDir;
	}
}
