<?php declare(strict_types = 1);

namespace Core\Model\Application;

use Core\Model\Http\Session;
use Core\Model\Sites;
use Navigations\Model\NavigationConfig;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\Http\Request;
use Nette\Http\Response;
use Nette\Utils\DateTime;

class AppState
{
	const CACHE_NAMESPACE = 'appState';

	protected Sites $sites;

	protected Session $session;

	protected Request $request;

	protected Response $response;

	protected Cache $cache;

	protected static array $state = [];

	public function __construct(Sites $sites, Session $session, Request $request, Response $response, Storage $storage)
	{
		$this->sites    = $sites;
		$this->session  = $session;
		$this->request  = $request;
		$this->response = $response;
		$this->cache    = new Cache($storage, self::CACHE_NAMESPACE);
	}

	public function init(): void
	{
		$domain = $this->sites->getCurrentSite()->getCurrentDomain();
		if (!$this->request->getCookie($this->getCountryCookieName()) && $domain->defaultCountry)
			$this->setCountry($domain->defaultCountry);
	}

	protected function getCountryCookieName(): string
	{
		$name   = 'country';
		$suffix = null;
		$domain = $this->sites->getCurrentSite()->getCurrentDomain();
		if (NavigationConfig::load('showDefaultLangInUrl') || !$domain->isDefault)
			$suffix = $domain->getLang();

		return $name . ($suffix ? ucfirst($suffix) : '');
	}

	public function setCountry(string $country): void
	{
		$key = $this->getCountryCookieName();
		$this->response->setCookie($key, $country, (new DateTime())->modify('+1 year'));
	}

	public function getCountry(): ?string { return $this->request->getCookie($this->getCountryCookieName()) ?: $this->sites->getCurrentSite()->getCurrentDomain()->defaultCountry; }

	public static function setState($key, $value)
	{
		self::$state[$key] = $value;
	}

	public static function getState($key, $default = null)
	{
		return self::$state[$key] ?? $default;
	}
}
