<?php declare(strict_types = 1);

namespace Core\Model;

use Core\Model\Entities\Country;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Dao;
use Nette\Caching\Cache;

/**
 * Class Countries
 * @package Core\Model
 *
 * @method Country|object|null getReference($id)
 */
class Countries extends BaseEntityService
{
	const CACHE_NAMESPACE = 'countries';

	/** @var string */
	protected $entityClass = Country::class;

	protected ?array $cOptionsForSelect = null;

	protected ?array $cDao = null;

	public function getCache(): Cache
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	public function get($id): ?Country
	{
		$countries = [];
		foreach ($this->getAll() as $country) {
			$countries[strtolower($country->getId())] = $country;
			$countries[strtoupper($country->getId())] = $country;
		}

		return $countries[$id] ?? null;
	}

	public function setPosition($id, $position)
	{
		if ($item = $this->get($id)) {
			$item->setPosition($position);
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	/** @return Country[]|null */
	public function getAll()
	{
		$countriesQuery = $this->getEr()->createQueryBuilder('c', 'c.id');
		$countriesQuery->orderBy('c.position');

		return $countriesQuery->getQuery()->getResult();
	}

	/**
	 * @return Dao\Country[]
	 */
	public function getDao(): array
	{
		if ($this->cDao === null) {
			$this->cDao = $this->getCache()->load(self::CACHE_NAMESPACE, function(&$dep) {
				$dep = [
					Cache::EXPIRATION => '1 month',
				];

				$data = [];
				foreach ($this->getEr()->createQueryBuilder('c')
					         ->orderBy('c.position')->getQuery()
					         ->getArrayResult() as $row) {
					$data[$row['id']] = new Dao\Country($row['id'], $row['iso3166_1'], $row['name']);
				}

				return $data;
			});
		}

		return $this->cDao;
	}

	public function getAllNameColumn(): array
	{
		if ($this->cOptionsForSelect === null) {
			$this->cOptionsForSelect = [];
			foreach ($this->getEr()->createQueryBuilder('c')->select('c.id, c.name')
				         ->orderBy('c.position')->getQuery()->getScalarResult() as $row) {
				$this->cOptionsForSelect[$row['id']] = $row['name'];
			}
		}

		return $this->cOptionsForSelect;
	}
}
