<?php declare(strict_types = 1);

namespace Core\Model\Templating\Filters;

use Core\Model\Helpers\Strings;
use Core\Model\SystemConfig;
use Currency\DI\CurrencyExtension;
use EshopOrders\Model\EshopOrdersConfig;
use Nette\Utils\Html;

class Price
{
	/** @var bool */
	protected $currencyInitWithFirstCall = false;

	/** @var callable[] */
	public static $priceInitCallback = [];

	/** @var bool */
	protected $currencyFromLeft = false;

	public static ?int $forceDecimals = null;

	/** @var string|null */
	protected $separator;

	/** @var string|null */
	protected $currency;

	/** @var string|null */
	protected $decSep;

	/** @var int|null */
	protected $decimals;

	/** @var array */
	public $otherCurrencies = [];

	/** @var array */
	protected $symbolToCode = [];

	/**
	 * Price constructor.
	 */
	public function __construct()
	{
		$config = SystemConfig::load('priceFilter');
		$this->setCurrency($config['currency']);
		$this->setSeparator($config['separator']);
		$this->setCurrencyFromLeft($config['currencyFromLeft']);
		$this->setDecimals($config['decimals']);
		$this->setDecSep($config['decSep']);
	}

	/**
	 * @param float       $price
	 * @param string|null $currency
	 * @param bool|null   $currencyFromLeft
	 * @param string|null $separator
	 * @param string|null $decSep
	 * @param int|null    $decimals
	 *
	 * @return string
	 */
	public function __invoke(?float $price, ?string $currency = null, ?bool $currencyFromLeft = null, ?string $separator = null, ?string $decSep = null, ?int $decimals = null): string
	{
		if ($price === null)
			return '';

		return $this->format($price, $currency, $currencyFromLeft, $separator, $decSep, $decimals);
	}

	/**
	 * @param float       $price
	 * @param string|null $currency
	 * @param bool|null   $currencyFromLeft
	 * @param string|null $separator
	 * @param string|null $decSep
	 * @param int|null    $decimals
	 *
	 * @return string
	 */
	public function format(float $price, ?string $currency = null, ?bool $currencyFromLeft = null, ?string $separator = null, ?string $decSep = null, ?int $decimals = null): string
	{
		if (!$this->currencyInitWithFirstCall)
			$this->callCurrencyCallbacks();

		$currency = $currency ?? $this->currency;

		if ($currencyFromLeft === null && $separator === null && $decSep === null && $this->getOtherCurrency($currency)) {
			$otherCurrency    = $this->getOtherCurrency($currency);
			$currencyFromLeft = $currencyFromLeft ?? $otherCurrency['currencyFromLeft'];
			$currency         = $decSep ?? $otherCurrency['symbol'];
			$decimals         = $decimals ?? $otherCurrency['decimals'];
		}

		$currencyFromLeft = $currencyFromLeft ?? $this->currencyFromLeft;
		$separator        = $separator ?? $this->separator;
		$decSep           = $decSep ?? $this->decSep;
		$decimals         = $decimals ?? $this->decimals;

		if (self::$forceDecimals)
			$decimals = self::$forceDecimals;

		return Strings::priceFormat($price, $currency, $separator, $currencyFromLeft, $decSep, $decimals);
	}

	/**
	 * @param bool $value
	 */
	public function setCurrencyFromLeft(bool $value = false): void
	{
		$this->currencyFromLeft = $value;
	}

	/**
	 * @param string|null $value
	 */
	public function setSeparator(?string $value = null): void
	{
		$this->separator = $value;
	}

	/**
	 * @param string|null $value
	 */
	public function setCurrency(?string $value = null): void
	{
		$this->currency = $value;
	}

	/**
	 * @param string|null $decSep
	 */
	public function setDecSep(?string $decSep = null): void
	{
		$this->decSep = $decSep;
	}

	/**
	 * @param int|null $decimals
	 */
	public function setDecimals(?int $decimals = null): void
	{
		$this->decimals = $decimals;
	}

	protected function callCurrencyCallbacks(): void
	{
		$this->currencyInitWithFirstCall = true;

		if (class_exists(CurrencyExtension::class) && !empty(self::$priceInitCallback)) {
			foreach (self::$priceInitCallback as $c)
				call_user_func($c, $this);
		}
	}

	public function addOtherCurrency(string $code, string $symbol, bool $currencyFromLeft, int $decimals): self
	{
		$this->otherCurrencies[$code] = [
			'code'             => $code,
			'symbol'           => $symbol,
			'currencyFromLeft' => $currencyFromLeft,
			'decimals'         => $decimals,
		];
		$this->symbolToCode[$symbol]  = $code;

		return $this;
	}

	public function getOtherCurrency(string $key): ?array
	{
		if (isset($this->otherCurrencies[$key]))
			return $this->otherCurrencies[$key];
		if (isset($this->otherCurrencies[$this->symbolToCode[$key]]))
			return $this->otherCurrencies[$this->symbolToCode[$key]];

		return null;
	}

	/**
	 * Vykrelsi 2 ceny pokud je i druha vyplnena
	 *
	 * @param float       $price1
	 * @param string      $code1
	 * @param float|null  $price2
	 * @param string|null $code2
	 * @param int|null    $decimals2
	 *
	 * @return Html
	 */
	public function renderTwoPrices(?float $price1, string $code1,
	                                ?float $price2 = null, ?string $code2 = null, ?int $decimals2 = null,
	                                ?float $priceWithoutVat = null, string $noVatText = ''): Html
	{
		if ($price1 === null)
			return Html::el();

		$wrap = Html::el('div class="price-value');

		$wrap->addHtml(Html::el('div class="price-value__price1')->setText($this->format($price1, $code1)));

		if ($price2)
			$wrap->addHtml(Html::el('div class="price-value__price2"')
				->setText($this->format($price2, $code2, null, null, null, $decimals2)));

		if (EshopOrdersConfig::load('orderForm.showPriceWithoutVat') && $priceWithoutVat)
			$wrap->addHtml(Html::el('div class="price-value__price3"')->setText($this->format($priceWithoutVat, $code1) . ' ' . $noVatText));

		return $wrap;
	}
}
