<?php declare(strict_types = 1);

namespace Core\Model\UI\Form\Controls;

use Core\Model\Helpers\Strings;
use Core\Model\UI\Form\Traits\TInputExtend;
use Nette\Forms\Controls\Checkbox;
use Nette\Forms\Form;
use Nette\Utils\Html;


class CheckboxInput extends Checkbox
{
	use TInputExtend;

	/**
	 * Generates a checkbox
	 * @return Html
	 */
	public function getControl(): Html
	{
		parent::getControl();

		$caption = $this->translate($this->caption);

		if (!$this->getIsMultiLanguage()) {
			return self::makeCheckbox($this->getHtmlName(), $caption, $this->value, false, $this->required,
				$this->disabled, $this->getHtmlId());
		} else {
			$wrap        = Html::el('div');
			$langs       = $this->getForm()->langsService->getLangs();
			$defaultLang = array_shift($langs);
			foreach ($langs as $k => $lang) {
				$getValue = $this->getValue();
				$value    = (is_array($getValue) ? $getValue[$k] : $getValue->$k) ?? '';

				$c = self::makeCheckbox($this->getHtmlName() . "[$k]", $caption, !empty($value), false, $this->required,
					$this->disabled, $this->getHtmlId() . "-$k");

				if ($defaultLang->getTag() != $lang->getTag()) {
					unset($c->attrs['required']);

					foreach ($c->attrs['data-nette-rules'] as $k2 => $rule) {
						if ($rule['op'] == ':filled')
							unset($c->attrs['data-nette-rules'][$k2]);
						break;
					}
				}

				$c->attrs['data-content-lang'] = $k;

				$wrap->addHtml($c);
			}

			return $wrap;
		}
	}

	public function getLabel($caption = null): void
	{
	}

	/**
	 * Makes a Bootstrap checkbox
	 *
	 * @param string      $name
	 * @param string|null $caption
	 * @param bool        $checked
	 * @param bool|mixed  $value pass false to omit
	 * @param bool        $required
	 * @param bool        $disabled
	 *
	 * @return Html
	 */
	public static function makeCheckbox($name, $caption = null, $checked = false, $value = false, $required = false, $disabled = false, $htmlId)
	{
		$wrap  = Html::el('div', ['class' => 'pretty p-check']);
		$label = Html::el('label', ['for' => $htmlId])
			->setText($caption);
		$input = Html::el('input', [
			'type'     => 'checkbox',
			'class'    => 'form-check-input',
			'name'     => $name,
			'disabled' => $disabled,
			'required' => $required,
			'checked'  => $checked,
			'id'       => $htmlId,
		]);

		if ($value !== false) {
			$input->attrs += [
				'value' => $value,
			];
		}

		$wrap->addHtml($input)
			->addHtml(
				Html::el('div class=state')
					->addHtml(Html::el('i class="icon fas fa-check"'))
					->addHtml($label)
			);

		return $wrap;

		$label->addHtml($input);
		$label->addText($caption);

		$line = Html::el('div', [
			'class' => 'form-check' . ($disabled ? ' disabled' : ''),
		]);
		$line->setHtml($label);

		return $line;
	}

	protected function getHttpData($type, $htmlTail = null)
	{
		if ($this->getIsMultiLanguage()) {
			$return = array_map(function($lang) use ($type, $htmlTail) {
				return $this->getForm()->getHttpData($type, $this->getHtmlName() . "[{$lang->getTag()}]" . $htmlTail);
			}, $this->getForm()->langsService->getLangs(false));

			return array_map(fn($val) => $val !== null ? '1' : null, $return);
		}
		else {
			return parent::getHttpData($type, $htmlTail);
		};
	}

	public function setValue($value)
	{
		if (!$this->getIsMultiLanguage()) {
			return parent::setValue($value);
		}

		$this->value = $value;

		return $this;
	}

}
