import autoComplete from "@tarekraafat/autocomplete.js/src/autoComplete";
import {addLoaderToElement} from "./Functions";

export class Autocomplete {
	constructor() {
		this.dataLoaded = [];
		this.lastXhr    = [];

		document.addEventListener('focusin', e => {
			let input = e.target;
			if (input.dataset.autocompleteUrl && input.autocompleteInited !== true) {
				input.autocompleteInited = true;

				const name = input.dataset.autocompleteName;
				let url    = input.dataset.autocompleteUrl;

				if (url.indexOf('/[data-content-lang]') !== -1) {
					const lang = input.dataset.contentLang || null;

					url = url.replace('/[data-content-lang]', lang ? '/' + lang : '')
				}

				let event = new CustomEvent('autocompleteOnPrepare', {
					detail: {
						name: name,
						url : url,
					},
				});
				document.dispatchEvent(event);

				url = event.detail.url;

				const inputId        = input.id;
				const resultItemData = input.dataset.resultItemData;
				const targetId       = input.dataset.autocompleteTarget;
				const target         = targetId ? document.getElementById(targetId) : null;
				let keys             = input.dataset.autocompleteKeys;

				if (keys) {
					keys = keys.split(',');
				} else {
					keys = ['id', 'name'];
				}

				input.parentElement.style.position = 'relative';

				let searchEngine = function (query, record) {
					try {
						return record && record.toString().toLowerCase().includes(query.toLowerCase())
							? record
							: null;
					} catch (e) {
					}

					return null;
				}

				const initAutoComplete = () => {
					new autoComplete({
						data        : {
							src   : this.dataLoaded[url],
							keys  : keys,
							cache : true,
							filter: (list) => {
								if (list.length < 2) {
									return list
								}

								const q = input.value.toLowerCase();
								return list.sort((a, b) => {
									const valA = a.match.toString().toLowerCase();
									const valB = b.match.toString().toLowerCase();

									const startsWithA = valA.startsWith(q);
									const startsWithB = valB.startsWith(q);

									if (startsWithA !== startsWithB) {
										return startsWithA ? -1 : 1;
									}

									const exactMatchA = valA === q;
									const exactMatchB = valB === q;

									if (exactMatchA !== exactMatchB) {
										return exactMatchA ? -1 : 1;
									}

									return 0;
								})
							}
						},
						searchEngine: searchEngine,
						selector    : '#' + inputId,
						threshold   : input.dataset.autocompleteThreshold ? input.dataset.autocompleteThreshold : 1,
						resultsList : {
							destination: '#' + input.id,
							maxResults : 20,
						},
						resultItem  : {
							element: (item, data) => {
								if (global.document.autocompleteResultItem && typeof global.document.autocompleteResultItem[name] === "function") {
									global.document.autocompleteResultItem[name](data, item);
								} else if (resultItemData) {
									let output = [];
									resultItemData.split(',').forEach(itm => {
										output.push(data.value[itm]);
									});

									item.innerHTML = output.join(' | ');
								} else {
									item.innerHTML = data.value.id + ' | ' + data.value.name;
								}
							},
						},
					});

					input.addEventListener('selection', e => {
						let feedback = e.detail;

						input.value = feedback.selection.value.name;
						if (target)
							target.value = feedback.selection.value.id;

						let event = new CustomEvent('autocompleteOnSelection', {
							detail: {
								name     : name,
								selection: feedback.selection,
								inputId  : inputId,
								targetId : targetId,
							},
						});
						Object.defineProperty(event, 'target', {writable: false, value: input});

						document.dispatchEvent(event);
					})
				}

				if (!this.dataLoaded[url]) {
					const loader = addLoaderToElement(input.parentElement);
					input.classList.add('autocomplete-loading');

					if (typeof this.lastXhr[url] !== 'undefined' && this.lastXhr[url].hasOwnProperty('abort')) {
						this.lastXhr[url].abort();
					}

					fetch(url, {
						method: 'GET',
					}).then(response => {
						return response.json();
					}).then(data => {
						this.dataLoaded[url] = [];

						for (let a in data) {
							let row = data[a];

							if (typeof row === 'string' || row instanceof String) {
								row = {
									id  : a,
									name: row,
								};
							} else {
								if (row.id === undefined) {
									row.id = a;
								}
							}

							this.dataLoaded[url][a] = row;
						}

						initAutoComplete();
						loader.remove();

						input.focus();
					}).catch(e => {
						loader.remove();
						input.value = 'error';
					});
				} else {
					initAutoComplete()
				}
			}
		})
	}
}
