<?php declare(strict_types = 1);

namespace Core\AdminModule\Components;

use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\DataToFormInputs;
use Core\Model\Settings;
use Core\Model\SystemConfig;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Exception;
use Nette\Utils\ArrayHash;
use Nette\Utils\Json;

abstract class SettingsForm extends BaseControl
{
	/** @var SystemConfig @inject */
	public $systemConfig;

	/** @var DataToFormInputs @inject */
	public $dataToFormInputs;

	/** @var Settings @inject */
	public           $settings;
	public string    $inputNamePrefix  = '';
	protected string $templateFile;
	public bool      $showLangSwitcher = true;

	public function __construct(
		protected ?string $namespace = null,
	)
	{
		$this->templateFile = __DIR__ . '/SettingsForm.latte';
	}

	public function render(): void
	{
		$this->template->thisForm = $this['form'];
		$this->template->setFile($this->templateFile);
	}

	/*******************************************************************************************************************
	 * ======================== COMPONENTS
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();
		$form->setShowLangSwitcher($this->showLangSwitcher);

		$form->addComponent(new BaseContainer(), 'component');

		$component = $form->getComponent('component');
		$parsed    = $this->dataToFormInputs->parseData($component, $this->getSystemConfigData());

		$form->onSuccess[] = $this->formSuccess(...);

		$this->template->componentStructure = $parsed['structure'];

		$defaults    = [];
		$settingsRaw = $this->settings->getAllRaw();
		foreach ($this->getSystemConfigData() as $v) {
			if (!isset($v['name'])) {
				continue;
			}

			$d = $settingsRaw[$v['name']] ?? null;
			if (Arrays::isJson($d) && !is_numeric($d)) {
				$d = Json::decode($d, Json::FORCE_ARRAY);
			}

			if ($v['type'] === 'text' && is_array($d)) {
				$d = Json::encode($d);
			}

			if ($d) {
				$defaults[$v['name']] = $d;
			}
		}

		$this->dataToFormInputs->setDefaults($form->getComponent('component'), $defaults);

		$form->addSaveCancelControl();
		$form->getComponent('saveControl')->showOnlySave = true;

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$this->em->beginTransaction();
		try {
			$vals = [];
			foreach ($form->getComponent('component')->getComponents() as $c) {
				if ($c instanceof BaseContainer) {
					$vals[$c->getName()] = Json::encode($c->getValues());
				} else if (is_object($c) && method_exists($c, 'getIsMultiLanguage') && $c->getIsMultiLanguage()) {
					/** @phpstan-ignore-next-line */
					$vals[$c->getName()] = Json::encode($c->getValue());
				} else {
					$vals[$c->getName()] = $c->getValue();
				}
			}

			$this->settings->saveMulti($vals);

			$this->presenter->flashMessageSuccess('default.saved');
			$this->presenter->redrawControl('flashes');

			$this->em->commit();
		} catch (Exception $e) {
			$this->em->rollback();
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}
		$this->settings->clearCache();
		$this->redrawControl('form');

		return true;
	}

	public function getDataForInputs(): array
	{
		return $this->namespace ? $this->systemConfig->get($this->namespace, []) : [];
	}

	/**
	 * @return array
	 */
	public function getSystemConfigData(): array
	{
		$result          = [];
		$inputNamePrefix = $this->inputNamePrefix ? $this->inputNamePrefix . '_' : '';
		foreach ($this->getDataForInputs() as $v) {
			if (isset($v['showIf']) && !$v['showIf']) {
				continue;
			}

			$arr = $v['name'];
			if (!is_array($v['name'])) {
				$arr = [$v['name']];
			}

			foreach ($arr as $name) {
				$tmp = $v;
				if (is_array($v['name'])) {
					$tmp['title'][0] = $v['title'][0] . '.' . $name;
				}

				if ($this->namespace !== 'globalData') {
					$tmp['name'] = $this->namespace . ucfirst((string) $name);
				}

				$tmp['name'] = $inputNamePrefix . $tmp['name'];
				$result[]    = $tmp;
			}
		}

		return $result;
	}
}
