<?php declare(strict_types = 1);

namespace Core\Components\FilesManager;

use Core\Model\FilesManager\FilesManager as FM;
use Core\Model\UI\BaseControl;
use Nette\Http\Request;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Utils\FileSystem;

class FilesManager extends BaseControl
{
	private static bool      $assetsLoaded = false;
	protected SessionSection $sessionSection;

	/** @var bool @persistent */
	public $modalShowed = false;

	/** @var string|null @persistent */
	public ?string $modalName = null;

	public ?array   $overrideRoots = null;
	protected bool  $isModal       = false;
	protected array $securedRoots  = [];

	public function __construct(
		protected FM      $fm,
		Session           $session,
		protected Request $request,
	)
	{
		$this->sessionSection = $session->getSection('filemanager');
	}

	public function render(): void
	{
		$this->template->uniqueId       = $this->getName();
		$this->template->assetsLoaded   = self::$assetsLoaded;
		$this->template->modalShowed    = $this->modalShowed;
		$this->template->isModal        = $this->isModal;
		$this->template->elFinderUrl    = $this->link('elFinder!');
		$this->template->closeModalLink = $this->link('closeModal!', [$this->name]);

		$this->template->render(__DIR__ . '/FilesManager.latte');
	}

	public function handleShowModal(string $modalName): void
	{
		$roots = $this->request->getPost('roots');
		if ($roots) {
			foreach ($roots as $root) {
				if (is_string($root)) {
					$root = ltrim($root, '/');

					$this->sessionSection->set('roots', [$modalName => [
						$root => [
							'driver'  => 'LocalFileSystem',
							'path'    => WWW_DIR . '/' . $root,
							'URL'     => '/' . $root,
							'tmbPath' => WWW_DIR . '/thumbs/' . $root,
							'tmbURL'  => '/thumbs/' . $root,
							'utf8fix' => true,],
					]]);
				} else {
					$this->sessionSection->set('roots', $root);
				}
			}
		}

		foreach ($this->securedRoots as $securedRoot) {
			$path = APP_DIR . '/' . $securedRoot;

			if (!file_exists($path)) {
				FileSystem::createDir($path);
			}

			$this->sessionSection->set('roots', [$modalName => [
				$securedRoot => [
					'driver'  => 'LocalFileSystem',
					'path'    => $path,
					'URL'     => '/' . $securedRoot,
					'tmbPath' => WWW_DIR . '/thumbs/' . $securedRoot,
					'tmbURL'  => '/thumbs/' . $securedRoot,
					'utf8fix' => true,
				],
			]]);
		}

		$this->setModalName($modalName);
		$this->modalShowed = true;

		if ($this->presenter->isAjax()) {
			$this->redrawControl('modal');
		} else {
			$this->redirect('this');
		}
	}

	public function handleCloseModal(string $modalName): void
	{
		$this->sessionSection->set('roots', [$modalName => []]);
	}

	public function handleElFinder(): void
	{
		$this->fm->overrideRoots = $this->overrideRoots;
		$this->fm->run(['rootsSession' => $this->modalName]);
	}

	public function setModal(bool $modal = true): void { $this->isModal = $modal; }

	public function setModalName(string $name): void { $this->modalName = $name; }

	public function addSecuredRoot(array $roots): void
	{
		foreach ($roots as $root) {
			$this->securedRoots[] = $root;
		}
	}
}
