<?php declare(strict_types = 1);

namespace Core\Console\Code;

use Doctrine\ORM\Tools\Console\Command\SchemaTool\AbstractCommand;
use Doctrine\ORM\Tools\SchemaTool;
use Exception;
use Nette\Utils\Finder;
use SplFileInfo;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use function str_contains;

#[AsCommand(
	name       : 'code:analyse',
	description: 'Analyse code by PHPStan Analytic Tool',
)]
class Analyse extends AbstractCommand
{
	protected const OPTION_PACKAGES_FULL = 'packgages', OPTION_PACKAGES_SHORTCUT = 'p';

	protected function configure(): void
	{
		$this->addOption(self::OPTION_PACKAGES_FULL, self::OPTION_PACKAGES_SHORTCUT, InputOption::VALUE_OPTIONAL);
	}

	protected function executeSchemaCommand(
		InputInterface  $input,
		OutputInterface $output,
		SchemaTool      $schemaTool,
		array           $metadatas,
		SymfonyStyle    $ui,
	): int
	{
		if (in_array(false, [
			file_exists(ROOT_DIR . '/vendor/phpstan/phpstan'),
			file_exists(ROOT_DIR . '/vendor/phpstan/phpstan-doctrine'),
			file_exists(ROOT_DIR . '/vendor/phpstan/phpstan-nette'),
		])) {
			$output->writeln(
				'<error>[ERROR] - Please add phpstan-doctrine and phpstan-nette as composer require-dev</error>',
			);

			return Command::FAILURE;
		}

		$output->writeln('START ANALYSE');
		$optionPackages = $input->getOption(self::OPTION_PACKAGES_FULL);
		$packages       = $optionPackages === null
			? []
			: (
			str_contains((string) $optionPackages, ',')
				? array_map('trim', explode(',', (string) $optionPackages))
				: [$optionPackages]
			);

		if (!$packages) {
			$packages = array_map(
				fn(SplFileInfo $info) => $info->getBasename(),
				iterator_to_array(Finder::findDirectories()->in(ROOT_DIR . '/vendor/pshkcms'), false),
			);
		}

		$exitCode = Command::SUCCESS;
		$packages = array_map('strtolower', $packages);

		try {
			foreach ($packages as $packageName) {
				$output->writeln('<info>[INFO] - ANALYSE ' . $packageName . '</info>');
				$packageSrcDir     = APP_DIR . '/../vendor/pshkcms/' . $packageName . '/src';
				$phpstanConfigFile = APP_DIR . '/../vendor/pshkcms/' . $packageName . '/phpstan.neon';

				if (!file_exists($phpstanConfigFile)) {
					$output->writeln(
						'<info>[INFO] - PACKAGE ' . $packageName . ' SKIPPED. phpstan.neon not found</info>',
					);
					$output->writeLn('');
					continue;
				}

				$phpstan = APP_DIR . '/../vendor/bin/phpstan';
				exec(
					sprintf(
						'php %s analyse %s -c %s --ansi --no-progress',
						$phpstan,
						$packageSrcDir,
						$phpstanConfigFile,
					),
					$out,
					$exitCode,
				);
				$output->writeLn($out);
			}
		} catch (Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');

			return Command::FAILURE;
		}

		return $exitCode;
	}
}
