<?php declare(strict_types = 1);

namespace Core\Model\Application;

use Core\Model\Countries;
use Core\Model\Http\Session;
use Core\Model\Sites;
use Core\Model\SystemConfig;
use Currency\Model\Config as CurrencyConfig;
use Currency\Model\Currencies;
use Navigations\Model\NavigationConfig;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\DI\Container;
use Nette\Http\Request;
use Nette\Http\Response;
use Nette\Utils\Arrays;
use Nette\Utils\DateTime;
use Nette\Utils\Json;

class AppState
{
	final public const CACHE_NAMESPACE = 'appState';
	protected Cache $cache;

	protected static array   $state           = [];
	protected static ?string $uniqueRequestId = null;
	protected static array   $jsAssets        = [];

	public function __construct(
		protected Sites     $sites,
		protected Session   $session,
		protected Request   $request,
		protected Response  $response,
		Storage             $storage,
		protected Countries $countries,
		protected Container $container,
	)
	{
		$this->cache = new Cache($storage, self::CACHE_NAMESPACE);
	}

	public function init(): void
	{
		if (php_sapi_name() === 'cli') {
			return;
		}

		$cookieBarEnabled = $this->container->getParameters()['system']['cookieBar']['enable'];
		$isCookieBarOpen  = false;
		if ($cookieBarEnabled) {
			$cookieBarKey    = $this->container->getParameters()['system']['cookieBar']['key'];
			$cookieBarValues = $this->request->getCookie($cookieBarKey);

			$allowCookieServices = [];
			$allowCookieSections = [];
			$isCookieBarOpen     = $cookieBarValues === null;
			/** @var array{type: string, services: array<string>, files: array<string>} $cookie */
			foreach ($cookieBarValues ? Json::decode($cookieBarValues, Json::FORCE_ARRAY) : [] as $cookie) {
				$allowCookieSections[] = $cookie['type'];
				foreach ($cookie['services'] as $service) {
					$allowCookieServices[] = $service;
				}
			}

			self::setState('allowCookieServices', $allowCookieServices);
			self::setState('allowCookieSections', $allowCookieSections);
		}
		self::setState('isCookieBarOpen', $isCookieBarOpen);

		$currentSite = $this->sites->getCurrentSite();

		$queryLang = $this->request->getQuery('lang');
		if ($queryLang) {
			$currentSite->currentLang = $queryLang;
		}

		$domain = $currentSite->getCurrentDomain();

		if (!$this->request->getCookie($this->getCountryCookieName()) && $domain->defaultCountry) {
			$this->setCountry($domain->defaultCountry);
		}

		if (
			$this->container->hasService('currency.currencies')
			&& class_exists('Currency\Model\Currencies')
			&& class_exists('Currency\Model\Config')
		) {
			/** @var Currencies $currencies */
			$currencies = $this->container->getService('currency.currencies');
			$currency   = $currencies->getCurrent();

			if (!empty(CurrencyConfig::load('siteAllowedCurrencies'))) {
				$allowedCurrencies = CurrencyConfig::load(
					'siteAllowedCurrencies.' . $domain->site->getIdent() . '.' . $domain->getLang(),
				);

				if (!empty($allowedCurrencies) && !Arrays::contains($allowedCurrencies, $currency->getCode())) {
					foreach (CurrencyConfig::load('siteAllowedCurrencies.' . $domain->site->getIdent()) ?? [] as $siteLang => $siteCurrencies) {
						if (Arrays::contains($siteCurrencies, $currency->getCode())) {
							header("Location: https://" . $currentSite->getDomains()[$siteLang]->getDomain() . '?curr=' . $currency->getCode());
							exit;
						}
					}

					$currency = $currencies->getActive()[$allowedCurrencies[0]];
					$currencies->setCurrentCurrency($currency->getId());
				}
			}

			self::setState('currency', $currency);

			if (CurrencyConfig::load('changeCountry') && count($currencies->getAll()) > 1) {
				$country = CurrencyConfig::load('changeCountry')[$currencies->getCurrent()->getCode()] ?? null;

				if ($country && $c = $this->countries->checkId($country)) {
					$this->setCountry($c);
				}
			}
		}
	}

	protected function getCountryCookieName(): string
	{
		$name   = 'country';
		$suffix = null;
		$domain = $this->sites->getCurrentSite()->getCurrentDomain();
		if (NavigationConfig::load('showDefaultLangInUrl') || !$domain->isDefault) {
			$suffix = $domain->getLang();
		}

		return $name . ($suffix ? ucfirst($suffix) : '');
	}

	public function setCountry(string $country): void
	{
		$key = $this->getCountryCookieName();

		if (!empty(SystemConfig::load('siteAllowedCountries', []))) {
			$domain           = $this->sites->getCurrentSite()->getCurrentDomain();
			$allowedCountries = SystemConfig::load(
				'siteAllowedCountries.' . $domain->site->getIdent() . '.' . $domain->getLang(),
			);

			if (!empty($allowedCountries) && !Arrays::contains($allowedCountries, $country)) {
				$country = $allowedCountries[0];
			}
		}

		self::setState('freshCountrySet', $country);
		$this->response->setCookie($key, $country, (new DateTime())->modify('+1 year'));
	}

	public function getCountry(): ?string
	{
		if (self::getState('freshCountrySet')) {
			return self::getState('freshCountrySet');
		}

		return $this->request->getCookie($this->getCountryCookieName()) ?: $this->sites->getCurrentSite()
			->getCurrentDomain()->defaultCountry;
	}

	public static function setState(string $key, mixed $value): void
	{
		self::$state[$key] = $value;
	}

	/**
	 * @return mixed|null
	 */
	public static function getState(string $key, mixed $default = null)
	{
		return self::$state[$key] ?? $default;
	}

	public static function getUniqueRequestId(string $salt = ''): string
	{
		if (self::$uniqueRequestId === null) {
			self::$uniqueRequestId = md5(uniqid($salt, true) . time());
		}

		return self::$uniqueRequestId;
	}

	public static function addJs(string $key, string $url): void { self::$jsAssets[$key] = $url; }

	public static function getJs(): array { return self::$jsAssets; }
}
