<?php declare(strict_types = 1);

namespace Core\Model\Helpers;

use function str_ends_with;
use function str_starts_with;

class DatabaseConfig
{
	public static array $config = [];

	protected static ?string $configKey = null;

	public static function getUser(): string
	{
		return (string) self::$config[self::getConfigKey()]['user'];
	}

	public static function getPassword(): string
	{
		return (string) self::$config[self::getConfigKey()]['password'];
	}

	public static function getDbName(): string
	{
		return (string) (self::$config[self::getConfigKey()]['dbname'] ?? self::$config['default']['dbname']);
	}

	public static function getHost(): string
	{
		return (string) (self::$config[self::getConfigKey()]['host'] ?? self::$config['default']['host']);
	}

	public static function getDsn(?string $dbName = null, ?string $host = null): string
	{
		$result = 'mysql:dbname=' . ($dbName ?: self::getDbName()) . ';';

		if (strtolower((string) PHP_OS) === 'linux' && self::$config[self::getConfigKey()]['host'] === null) {
			$host = '/var/lib/mysql/mysql.sock';
		} else {
			$host = ($host ?: self::$config[self::getConfigKey()]['host']) ?: '127.0.0.1';
		}

		if (str_ends_with($host, '.soc')) {
			$result .= 'unix_socket=' . $host;
		} else {
			$result .= 'host=' . $host;
		}

		return $result;
	}

	public static function getConfigKey(): string
	{
		$requestUri = (string) $_SERVER['REQUEST_URI'];
		$httpHost   = (string) $_SERVER['HTTP_HOST'];

		if (is_null(self::$configKey)) {
			if ((php_sapi_name() === 'cli' || str_starts_with($requestUri, '/cron'))
				&& isset(self::$config['cron']['user'])
				&& isset(self::$config['cron']['password'])) {
				self::$configKey = 'cron';
			} else if ((str_starts_with($requestUri, '/admin') || str_starts_with($httpHost, 'admin.'))
				&& isset(self::$config['admin']['user'])
				&& isset(self::$config['admin']['password'])) {
				self::$configKey = 'admin';
			} else {
				foreach (self::$config['urlPrefix'] ?? [] as $prefix => $v) {
					if (str_starts_with(ltrim($requestUri, '/'), (string) $prefix)) {
						self::$configKey = $v;
						break;
					}
				}

				if (self::$configKey === null) {
					self::$configKey = 'default';
				}
			}
		}

		return self::$configKey;
	}
}
