<?php declare(strict_types = 1);

namespace Core\Model\Images;

use Core\Model\Helpers\Strings;
use Core\Model\MathParser\Math;
use Core\Model\Sites;
use Core\Model\SystemConfig;
use Exception;
use Nette;
use Nette\Utils\Image as NImage;
use function str_ends_with;

class ImagePipe
{
	use Nette\SmartObject;

	protected string  $cacheDir;
	protected string  $wwwDir;
	protected ?string $path           = null;
	protected ?array  $watermark      = null;
	protected string  $originalPrefix = "original";
	protected string  $baseUrl;
	protected ?string $namespace      = null;

	/** @var array */
	public $onBeforeSaveThumbnail = [];

	/**
	 * @param array $params
	 */
	public function __construct($params, Nette\Http\Request $httpRequest, protected Sites $sites)
	{
		if (php_sapi_name() === 'cli') {
			if (isset($params['wwwDir'])) {
				$params['wwwDir'] = WWW_DIR;
			}

			if (isset($params['cacheDir'])) {
				$params['cacheDir'] = WWW_DIR . '/thumbs';
			}
		}
		$this->wwwDir    = $params['wwwDir'] ?? WWW_DIR;
		$this->cacheDir  = $params['cacheDir'] ?? WWW_DIR . '/thumbs';
		$this->watermark = $params['watermark'] ?? null;
		$this->baseUrl   = rtrim($httpRequest->url->baseUrl, '/');
	}

	public function setBaseUrl(string $url): void
	{
		$this->baseUrl = rtrim($url, '/');
	}

	public function setPath(string $path): void { $this->path = $path; }

	public function setCacheDir(string $dir): void { $this->cacheDir = $dir; }

	public function getCacheDir(): string { return $this->cacheDir; }

	public function getPath(): string
	{
		return $this->path ?? $this->baseUrl . str_replace($this->wwwDir, '', $this->cacheDir);
	}

	public function setOriginalPrefix(string $originalPrefix): void { $this->originalPrefix = $originalPrefix; }

	public function getOriginalPrefix(): string { return $this->originalPrefix; }

	public function setNamespace(string $namespace): self
	{
		if (empty($namespace)) {
			$this->namespace = null;
		} else {
			$this->namespace = $namespace . "/";
		}

		return $this;
	}

	protected function canUseWebP(int $width, int $height): bool
	{
		return $width > 45 && $height > 45;
	}

	public function request(
		?string         $image,
		?string         $size = null,
		int|string|null $flags = null,
		?bool           $watermark = false,
		?bool           $strictMode = false,
		?string         $outputExtension = null,
		?int            $quality = null,
	): string
	{
		if ($strictMode === null) {
			$strictMode = false;
		}

		if (!$image) {
			return '#';
		}

		$extension = pathinfo((string) explode('?', (string) $image)[0], PATHINFO_EXTENSION);

		if ($extension === 'svg' || !in_array(strtolower($extension), ['jpg', 'jpeg', 'png', 'gif', 'webp', 'bmp'])) {
			return $image;
		}

		$remoteUrl = null;
		if (str_starts_with($image, 'http')) {
			$remoteUrl = new Nette\Http\Url($image);
		}

		$image = ltrim($image, '/');
		if (empty($image)) {
			return "#";
		}

		if ($size === null) {
			return $image;
		}

		[$width, $height] = explode("x", $size);
		$width  = (int) $width;
		$height = (int) $height;

		if ($width === 0 && $height === 0) {
			return $image;
		}

		$cropX = '50%';
		$cropY = '50%';

		if ($flags == null) {
			$flags = NImage::FIT;
		} else if (strpos((string) $flags, 'crop') === 0) {
			$crop = ltrim((string) $flags, 'crop');
			[$cropX, $cropY] = explode('x', $crop);
			$cropX = (string) $cropX;
			$cropY = (string) $cropY;
			$flags = 'crop';
		} else if (!is_int($flags)) {
			switch (strtolower($flags)) {
				case "fit":
					$flags = NImage::FIT;
					break;
				case "fill":
					$flags = NImage::FILL;
					break;
				case "exact":
					$flags = NImage::EXACT;
					break;
				case "shrink_only":
					$flags = NImage::SHRINK_ONLY;
					break;
				case "stretch":
					$flags = NImage::STRETCH;
					break;
			}
		}

		$f = "{$flags}_{$width}x{$height}" . ($watermark ? '_wm' : '') . ($quality !== null ? sprintf(
				'_q%s',
				$quality,
			) : '');
		if ($cropX != '50%' && $cropY != '50%') {
			$f .= '_' . (Strings::webalize((string) $cropX)) . 'x' . (Strings::webalize((string) $cropY));
		}

		if ($remoteUrl) {
			$dirname = ltrim((string) dirname($remoteUrl->getPath()), '/');
			$image   = (string) explode('?', (string) $image)[0];
		} else {
			$dirname = dirname($image);
		}
		$thumbFile    = $this->cacheDir . "/" . $dirname . "/{$f}/" . basename($image);
		$originalFile = $remoteUrl ? $remoteUrl->getAbsoluteUrl() : $this->wwwDir . "/" . $image;

		if ($outputExtension) {
			$info      = pathinfo($thumbFile);
			$thumbFile = $info['dirname'] . DS . $info['filename'] . '.' . $outputExtension;
			$thumbPath = str_replace($this->cacheDir, '', $thumbFile);
		} else if ($this->canUseWebP($width, $height) && ImageHelper::supportWebP()) {
			$info      = pathinfo($thumbFile);
			$thumbFile = $info['dirname'] . DS . substr($info['filename'], 0, 150) . '.webp';
			$thumbPath = str_replace($this->cacheDir, '', $thumbFile);
		} else {
			$thumbPath = str_replace($this->cacheDir, '', $thumbFile);
		}

		if (!file_exists($thumbFile)) {
			if (!self::mkdir(dirname($thumbFile))) {
				return $originalFile;
			}

			if (file_exists($originalFile) || $remoteUrl) {
				if (str_contains((string) mime_content_type($originalFile), 'empty')) {
					return $image;
				}

				set_time_limit(60);

				if (str_ends_with($originalFile, '_jpg')) {
					$originalFile = str_replace('_jpg', '.jpg', $originalFile);
				}

				try {
					$img = NImage::fromFile($originalFile);
				} catch (Exception) {
					return $image;
				}

				if ($this->canUseWebP($width, $height) && ImageHelper::supportWebP() && pathinfo(
						$originalFile,
						PATHINFO_EXTENSION,
					) === 'png') {
					$img->paletteToTrueColor();
					$img->alphaBlending(false);
					$img->saveAlpha(true);
				}

				if ($flags === "crop") {
					$imgWidth  = $img->getWidth();
					$imgHeight = $img->getHeight();

					if ($imgWidth > $imgHeight) {
						$cropWidth  = (int) round(($imgHeight / 100) * ($width / $height) * 100, 0);
						$cropHeight = $imgHeight;
					} else if ($imgWidth < $imgHeight) {
						$cropWidth  = $imgWidth;
						$cropHeight = (int) round(($imgWidth / 100) * ($height / $width) * 100, 0);
					} else {
						$cropWidth  = $imgWidth;
						$cropHeight = $imgHeight;
					}

					$img->crop($cropX, $cropY, $cropWidth, $cropHeight)->resize($width, $height, NImage::EXACT);
					$img->sharpen();
				} else {
					$img->resize($width, $height, (int) $flags | NImage::SHRINK_ONLY);
				}

				if ($watermark) {
					$this->addWatermark($img);
				}

				$this->onBeforeSaveThumbnail($img, $this->namespace, $image, $width, $height, $flags);
				$img->save($thumbFile, $quality);
			} else if ($strictMode) {
				return $originalFile;
			}
		}
		$this->namespace = null;
		$thumbPath       = str_replace(DS, '/', $thumbPath);

		return str_replace(' ', '%20', $this->getPath() . $thumbPath);
	}

	/**
	 * @param string $dir
	 *
	 * @return bool
	 */
	private static function mkdir($dir)
	{
		$oldMask = umask(0);
		@mkdir($dir, 0777, true);
		@chmod($dir, 0777);
		umask($oldMask);

		if (!is_dir($dir) || !is_writable($dir))
			return false;

		return true;
	}

	public function removeThumbs(string $file): void
	{
		$dir = str_replace($this->wwwDir, '', $file);
		$dir = pathinfo($dir, PATHINFO_DIRNAME);

		foreach (glob($this->cacheDir . $dir . '/**/' . basename($file)) ?: [] as $v) {
			@unlink($v);
		}
	}

	protected function addWatermark(NImage $img): void
	{
		if (!isset($this->watermark['file'], $this->watermark['size'], $this->watermark['left'], $this->watermark['top']) || !file_exists(
				$this->watermark['file'],
			)) {
			return;
		}

		$watermark = NImage::fromFile($this->watermark['file']);
		$math      = new Math();

		$size  = str_replace('$imgWidth', (string) $img->getWidth(), (string) $this->watermark['size']);
		$width = $math->evaluate($size);
		$width = is_float($width) ? ((int) $width) : $width;
		$watermark->resize($width, 0, NImage::FIT);

		$left = str_replace(['$imgWidth', '$wmWidth', ','], [$img->getWidth(), $width,
			'.'], (string) $this->watermark['left']);
		$left = $math->evaluate($left);
		$top  = str_replace(['$imgHeight', '$wmHeight', ','],
			[$img->getHeight(), $watermark->getHeight(), '.'],
			(string) $this->watermark['top']);
		$top  = $math->evaluate($top);

		$img->alphaBlending(true);
		$watermark->alphaBlending(true);
		$img->place(
			$watermark,
			is_float($left) ? ((int) $left) : $left,
			is_float($top) ? ((int) $top) : $top,
			$this->watermark['opacity'] ?? 80,
		);
	}

	public function generatePlaceholder(int $width = null, int $height = null): string
	{
		if (!SystemConfig::load('images.placeholderOverlay.allow') || !$width || !$height) {
			return '';
		}

		$width  = max(1, $width);
		$height = max(1, $height);

		$ext = 'png';
		if ($this->canUseWebP($width, $height) && ImageHelper::supportWebP()) {
			$ext = 'webp';
		}

		$overlayLogo = SystemConfig::load(
			'images.placeholderOverlay.bySite.' . $this->sites->getCurrentSite()
				->getIdent() . '.logo',
		)
			?: SystemConfig::load('images.placeholderOverlay.logo');

		$bgColors = SystemConfig::load(
			'images.placeholderOverlay.bySite.' . $this->sites->getCurrentSite()
				->getIdent() . '.bgColors',
		)
			?: SystemConfig::load('images.placeholderOverlay.bgColors');

		if ($overlayLogo === 'currentSite') {
			$overlayLogo = $this->sites->getCurrentSite()->logo;
		}

		$logoHash = md5($overlayLogo . serialize($bgColors));

		$file      = "/thumbs/placeholders/{$logoHash}-{$width}x{$height}." . $ext;
		$thumbFile = WWW_DIR . $file;

		if (file_exists($thumbFile)) {
			return $file;
		}

		$image = NImage::fromBlank($width, $height, $bgColors);

		try {
			if ($overlayLogo && file_exists(WWW_DIR . $overlayLogo)) {
				$overlay       = NImage::fromFile(WWW_DIR . $overlayLogo);
				$overlayWidth  = intval($width * .55);
				$overlayHeight = intval($height * .55);
				$overlay->resize($overlayWidth, $overlayHeight, NImage::FIT | NImage::SHRINK_ONLY);

				$image->place(
					$overlay,
					intval($width / 2 - $overlay->getWidth() / 2),
					intval($height / 2 - $overlay->getHeight() / 2),
					25,
				);
			}
		} catch (Exception) {
		}

		Nette\Utils\FileSystem::createDir(dirname($thumbFile));

		$image->save($thumbFile);

		$file = str_replace(DS, '/', $file);

		return $file;
	}
}
