<?php declare(strict_types = 1);

namespace Core\Model\TemplateReader;

use Contributte\Translation\Translator;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\DataToFormInputs;
use Core\Model\Helpers\Strings;
use Core\Model\Images\ImagePipe;
use Core\Model\Module;
use Core\Model\SystemConfig;
use Core\Model\TemplateReader\Entity\ITemplate;
use Core\Model\TemplateReader\Entity\ITemplate as EntityITemplate;
use Core\Model\TemplateReader\Providers\ITemplateTextType;
use Core\Model\Templating\Macros\TemplateText;
use Core\Model\Templating\Template;
use Core\Model\Templating\TemplateFactory;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\Controls\CheckboxListInput;
use Core\Model\UI\Form\Controls\HiddenInput;
use Core\Model\UI\Form\Controls\SelectInput;
use Nette\Application\UI;
use Nette\Forms\IControl;
use Nette\Http\Request;
use Nette\Utils\FileSystem;
use Nette\Utils\Json;
use Pages\Model\Paths;
use Pages\Model\TemplateTextType\Block;
use Pages\Model\TemplateTextType\Image;

class TemplateReader
{
	public ?UI\Control $uiControl = null;

	protected TemplateTextTypesCollection $templateTextTypesCollection;
	protected Request                     $httpRequest;
	protected TemplateFactory             $templateFactory;
	protected ImagePipe                   $imagePipe;
	protected Translator                  $translator;
	protected DataToFormInputs            $dataToFormInputs;
	protected Paths                       $pathsService;

	//TODO typ šablony (místo default proměnná)
	protected ?string $templatesDir       = null;
	protected string  $translateKey       = 'templateReader';
	protected string  $tmpDir             = TMP_DIR . '/cache/latte/templateReader';
	protected array   $componentStructure = [];

	public static string   $mode             = 'pages';
	protected static array $createdTemplates = [];

	protected array $cRawData = [];

	public function __construct(
		TemplateFactory             $templateFactory,
		TemplateTextTypesCollection $templateTextTypesCollection,
		Request                     $request,
		ImagePipe                   $imagePipe,
		Translator                  $translator,
		DataToFormInputs            $dataToFormInputs,
		Paths                       $paths
	)
	{
		$this->templateTextTypesCollection = $templateTextTypesCollection;
		$this->httpRequest                 = $request;
		$this->templateFactory             = $templateFactory;
		$this->imagePipe                   = $imagePipe;
		$this->translator                  = $translator;
		$this->dataToFormInputs            = $dataToFormInputs;
		$this->pathsService                = $paths;
		$this->reInitComponentStructure();
	}

	/**
	 * Vrátí strukturu komponent v poli pro vykreslení
	 */
	public function getComponentStructure(): array
	{
		return $this->componentStructure;
	}

	/**
	 * Nastavení klíče pro překlad (admin.key.hodnota)
	 */
	public function setTranslateKey(string $key): void
	{
		$this->translateKey = $key;
	}

	/**
	 * Nastaví výchozí stav struktury pro znovuvytvoření
	 */
	private function reInitComponentStructure(): void
	{
		$this->componentStructure = ['sections' => []];
	}

	/**
	 * Nastavení složky ze kterých se načtou šablony
	 */
	public function setTemplatesDir(string $dir): void
	{
		$this->templatesDir = $dir;
	}

	public function getTemplates(?string $keyPrefix = null): array
	{
		$templates = [];
		foreach (glob($this->templatesDir . '/*.latte') ?: [] as $file) {
			$fileName = pathinfo($file, PATHINFO_FILENAME);
			$tKey     = 'admin.' . $this->translateKey . '.' . $fileName;
			$t        = $this->translator->translate($tKey);

			$k = basename($file);
			if ($keyPrefix) {
				$k = $keyPrefix . '|' . $k;
			}

			$templates[$k] = $t === $tKey ? $fileName : $t;
		}

		if (!empty($templates)) {
			asort($templates, SORT_LOCALE_STRING);
		}

		return $templates;
	}

	/**
	 * Vrátí složku kam se ukládají temp soubory při vytváření struktury
	 */
	protected function getTmpDir(): string
	{
		if ($this->templatesDir) {
			return $this->tmpDir . DS . basename($this->templatesDir);
		}

		return $this->tmpDir;
	}

	/**
	 * Vrrátí složku kde jsou šablony
	 */
	public function getTemplatePath(string $template): string
	{
		return $this->templatesDir . DS . $template;
	}

	/**
	 * TODO stejné jako v DataToFormInputs. Rozdil jenom v $this->componentStruture
	 *
	 * Přečte nastavenou šablonu a do komponenty formuláře načte potřebné komponenty
	 */
	public function loadTemplateComponents(BaseContainer $container, string $file, array $texts): void
	{
		try {
			/** @var UI\Control|null $control */
			$control         = $container->lookup(UI\Control::class);
			$this->uiControl = $control;
		} catch (\Exception $e) {
			/** @var UI\Control|null $control */
			$control         = $container->lookup(null);
			$this->uiControl = $control;
		}

		$this->reInitComponentStructure();

		$result = $this->dataToFormInputs->parseData($container, $this->getTemplateInputs($file), $texts);
		if (isset($result['structure'])) {
			$this->componentStructure = $result['structure'];
		}
	}

	/**
	 * Nastaví výchozí hodnoty formuláři se zvolenou šablonou
	 *
	 * @param array|EntityITemplate $data
	 */
	public function setDefaults(BaseContainer $container, $data, string $template, ?string $file = null): void
	{
		/** @var IControl[] $components */
		$components       = $container->components;
		$availableLocales = $this->translator->getLocalesWhitelist();

		if ($data instanceof EntityITemplate === false) {
			$tmp = [];

			foreach ($data as $l => $v) {
				if (!in_array($l, $availableLocales)) {
					$tmp[$l] = $v;
					continue;
				}

				$multiLang   = [];
				$noMultiLang = [];
				foreach ($components as $name => $c) {
					if (
						($c instanceof BaseContainer && $c->getCustomData('multiLang') === true)
						|| (method_exists($c, 'getIsMultiLanguage') && $c->getIsMultiLanguage())
					) {
						if (is_array($v)) {
							$tmp[$name][$l] = $v[$name];
						} else if (is_string($v)) {
							if (count($multiLang) > 0) {
								continue;
							}
							$tmp[$name][$l] = $v;
							$multiLang[]    = $name;
						} else {
							$tmp[$name][$l] = $v->getText($name) ?? null;
						}
					} else {
						if (is_array($v)) {
							$tmp[$name] = $v[$name];
						} else if (is_string($v)) {
							if (count($noMultiLang) > 0) {
								continue;
							}
							$tmp[$name]    = $v;
							$noMultiLang[] = $name;
						} else {
							$tmp[$name] = $v->getText($name) ?? null;
						}
					}
				}
			}

			$data = $tmp;
		}

		if (
			!$file
			|| ($data instanceof EntityITemplate && $template === $file)) {
			foreach ($components as $k => $c) {
				if ($data instanceof EntityITemplate) {
					$value = $data->getText($k);

					if ($value === '') {
						$tmp = [];
						foreach ($data->getTexts() as $textsK => $textsV) {
							if (method_exists($c, 'getName') && Strings::startsWith($textsK, $c->getName() . '_')) {
								$tmp[$textsK] = $textsV;
							}
						}

						if (!empty($tmp)) {
							$value = $tmp;
						}
					}
				} else {
					$value = $data[$k] ?? null;
				}

				try {
					if (method_exists($c, 'getIsMultiLanguage') && (bool) $c->getIsMultiLanguage() && (!is_array($value) || !isset($value[$this->translator->getDefaultLocale()]))) {
						$valueTmp = [];
						foreach ($this->translator->getLocalesWhitelist() as $l) {
							$valueTmp[$l] = $value;
						}

						$value = $valueTmp;
					}
				} catch (\Exception $e) {
				}

				if ($c instanceof SelectInput) {
					if (is_array($value)) {
						$c->setDefaultValue($value);
					} else if (array_key_exists($value, $c->getItems())) {
						$c->setDefaultValue($value);
					}
				} else if ($c instanceof CheckboxListInput) {
					// u dynamickeho modulu se data pro checkbox ukladaji jinak, pro tato uprava
					if (!$value && self::$mode === 'dynamicModule') {
						$fullKey = $k . '_';

						if ($data instanceof ITemplate) {
							$texts = $data->getTexts();
						} else {
							$texts = $data;
						}

						$value = Arrays::search($texts, static function($key, $val) use ($fullKey) { return Strings::startsWith($key, $fullKey); }, false);
					}

					$tmp = [];
					foreach ($value as $v) {
						if (array_key_exists($v, $c->getItems())) {
							$tmp[] = $v;
						}
					}

					$c->setDefaultValue($tmp);
				} else if ($c instanceof HiddenInput) {
					$c->setValue($value);
				} else if ($c instanceof \Nette\Forms\Controls\BaseControl) {
					if ((!is_array($value) && !empty($value)) || (is_array($value) && Arrays::some($value, fn($v) => !empty($v)))) {
						$c->setDefaultValue($value);
					}
				} else if ($c instanceof BaseContainer) {
					if ($c->getCustomData('ttType') === Image::class && $c->getCustomData('multiLang') !== true && (is_object($value) || is_array($value))) {
						$c->setDefaults($value);
					} else {
						if (Arrays::isJson($value)) {
							$value = Json::decode($value, Json::FORCE_ARRAY);
						}

						$this->setDefaults($c, $value, $template, $file);
					}
				}
			}
		}
	}

	/**
	 * Zpracuje zvolenou šablonu a vrátí pole s připravenýma hodnotama
	 */
	public function getTemplateInputs(string $template): array
	{
		$file = (string) $this->getTemplatePath($template);

		if (!isset(self::$createdTemplates[$file])) {
			if (SystemConfig::load('useLatteTemplateReader')) {
				$fileForTemplate = $file;
			} else {
				$tmpFile     = $this->getTmpDir() . DS . basename($file);
				$saveToCache = [];
				$re          = '/{(include|import) (.*)}/m';

				$content = @file_get_contents($file);

				if (!$content) {
					return [];
				}

				FileSystem::createDir($this->getTmpDir());

				// Vyhledání všech souborů které se načítají
				$getFileContent = function($content) use (&$getFileContent, &$saveToCache, $re, $file) {
					preg_match_all($re, $content, $matches, PREG_SET_ORDER, 0);

					foreach ($matches as $match) {
						/** @phpstan-ignore-next-line */
						if (!isset($match[2])) {
							continue;
						}

						$fileToInclude = null;
						$match[2]      = explode(',', $match[2])[0];

						if ((strpos($match[2], "'") !== 0 && strpos($match[2], "'") !== 0)) {
							ob_start();
							eval("echo  $match[2];");
							$match[2] = ob_get_clean();
							$match[2] = addslashes((string) $match[2]);
							$match[2] = (string) iconv('ASCII', 'UTF-8//IGNORE', $match[2]);
						} else {
							$match[2] = trim($match[2], '\'"');
						}

						if (file_exists($match[2])) {
							$fileToInclude = $match[2];
						} else if (file_exists(dirname($match[2]) . $match[2])) {
							$fileToInclude = dirname($match[2]) . $match[2];
						} else if (file_exists(dirname($match[2]) . DS . $match[2])) {
							$fileToInclude = dirname($match[2]) . DS . $match[2];
						} else if (file_exists(dirname($file) . DS . $match[2])) {
							$fileToInclude = dirname($file) . DS . $match[2];
						}

						if (isset($fileToInclude)) {
							$fileToInclude = (string) $fileToInclude;

							$fileContent = $getFileContent(file_get_contents($fileToInclude));
							$fileContent = preg_replace("/{define(.*)}/", "", $fileContent);
							$fileContent = str_replace("{/define}", '', $fileContent);

							$tmpFileToInclude = $this->getTmpDir() . DS . pathinfo($file, PATHINFO_FILENAME) . DS . basename(dirname($fileToInclude)) . DS . basename($fileToInclude);
							$tmpFileToInclude = addslashes($tmpFileToInclude);
							$tmpFileToInclude = (string) iconv('ASCII', 'UTF-8//IGNORE', $tmpFileToInclude);

							$content = str_replace((string) $match[0], '{incldTemp "' . $tmpFileToInclude . '"}', $content);
							FileSystem::createDir(dirname($tmpFileToInclude));
							$saveToCache[$tmpFileToInclude] = $fileContent;
							//										$content = str_replace($match[0], $fileContent, $content);
							$content = $getFileContent($content);
						} else {
							$content = str_replace((string) $match[0], '', $content);
						}
					}

					return $content;
				};

				$content               = $getFileContent(file_get_contents($file));
				$saveToCache[$tmpFile] = $content;

				// Upravení obsahu pro použití v adminu a uložení do temp
				foreach ($saveToCache as $f => $content) {
					$re      = '/{(templateText|tt) (.*)}/m';
					$content = preg_replace($re, "R$0", $content);
					$content = str_replace('{incldTemp ', '{include ', $content);
					$content = str_replace('R{tt', '{ttR', $content);
					$content = str_replace('R{templateText', '{templateTextR', $content);
					$content = preg_replace("/{link(.*)}/", "#", $content);
					$content = preg_replace("/{snippet(.*)}/", "", $content);
					$content = preg_replace("/{\/snippet}/", "", $content);
					$content = preg_replace("/n:snippet=\"(.*)\"/", "", $content);

					// TODO přidat podporu pro komponenty
					$content = preg_replace("/{control(.*)}/", "#", $content);

					file_put_contents((string) $f, $content);
				}

				file_put_contents($tmpFile, $content);
				$fileForTemplate = $tmpFile;
			}
			// Vytvoření čísté šablony pro čtení
			$inputs = [];

			if (Module::isAdmin()) {
				$this->templateFactory->onCreate[] = static function(\Nette\Bridges\ApplicationLatte\Template $template) {
					$template->getLatte()->setTempDirectory(null);
				};
			}

			$latte = $this->templateFactory->createTemplate($this->uiControl, null, false);
			$latte->setFile($fileForTemplate);
			$content = $latte->renderToString();

			$re = '/{(templateText|tt) (.*)}/m';
			preg_match_all($re, $content, $output, PREG_SET_ORDER, 0);

			foreach ($output as $v) {
				/** @phpstan-ignore-next-line */
				if (!isset($v[2])) {
					continue;
				}

				$arr = TemplateText::parseInputAttrs($v[2]);

				if (isset($arr['name']) && isset($arr['title'])) {
					foreach ($arr as $attrKey => $attrValue) {
						// Připravení dat pro překlad
						if (in_array($attrKey, ['title', 'sectionName', 'groupName'])) {
							$tmp = explode(', ', $attrValue, 2);

							if (isset($tmp[1])) {
								$params = [];
								foreach (explode(',', $tmp[1]) as $param) {
									$param = trim($param);

									[$k, $v] = explode('=>', $param);
									$params[$k] = $v;
								}
								$tmp[1] = $params;
							}

							$arr[$attrKey] = $tmp;
						}
					}

					$inputs[] = $arr;
				}
			}

			self::$createdTemplates[$file] = $inputs;
		}

		return self::$createdTemplates[$file];
	}

	public function readRawData(?array $data = null, ?array $inputs = null): ?array
	{
		if (!$data || !$inputs) {
			return $data;
		}

		$key = md5(serialize($data));
		if (array_key_exists($key, $this->cRawData)) {
			return $this->cRawData[$key];
		}

		foreach ($inputs as $input) {
			$value = $data[$input['name']] ?? null;
			/** @var ITemplateTextType $tt */
			$tt = $this->templateTextTypesCollection->getItemByType($input['type']);

			if ($tt instanceof Block) {
				continue;
			}

			$tt->setDefault($value);

			$result = $tt->render($input);

			if (is_array($result) && array_key_exists($this->translator->getLocale(), $result)) {
				$result = $result[$this->translator->getLocale()];
			}

			$data[$input['name']] = $result;
		}

		$this->cRawData[$key] = $data;

		return $data;
	}

	/**
	 * @param object|Template $v
	 *
	 * @return mixed|null
	 */
	public function getTemplateTextValues($v)
	{
		$loop = static function($v2) use (&$loop) {
			/** @var object|Template $v2 */
			if (property_exists($v2, 'templateTextValues')) {
				return $v2->templateTextValues;
			} else if (property_exists($v2, 'uiControl')) {
				if (property_exists($v2->uiControl, 'template')) {
					return $loop($v2->uiControl->template);
				} else if (is_object($v2->uiControl) && method_exists($v2->uiControl, 'getTemplate')) {
					return $loop($v2->uiControl->getTemplate());
				}
			}

			return null;
		};

		return $loop($v);
	}
}
