<?php declare(strict_types = 1);

namespace Core\Model\Templating\Macros;

use Core\Model\Helpers\Arrays;
use Core\Model\Module;
use Core\Model\SystemConfig;
use Latte\Compiler;
use Latte\MacroNode;
use Latte\Macros\MacroSet;
use Latte\PhpWriter;
use Nette\Utils\Json;
use function str_ends_with;
use function str_starts_with;

class TemplateText extends MacroSet
{
	public static function install(Compiler $compiler): void
	{
		$self = new self($compiler);
		$self->addMacro('templateText', $self->macroTemplateText(...));
		$self->addMacro('tt', $self->macroTemplateText(...));
		$self->addMacro('templateTextR', $self->macroTemplateTextR(...));
		$self->addMacro('ttR', $self->macroTemplateTextR(...));
	}

	public function macroTemplateText(MacroNode $node, PhpWriter $writer): string
	{
		if (SystemConfig::load('useLatteTemplateReader') && Module::isAdmin()) {
			return $this->macroTemplateTextR($node, $writer);
		}

		$oData = self::parseInputAttrs($node->args);

		if (!isset($oData['name'])) {
			return '';
		}

		if (isset($oData['toVar'])) {
			$str = '${"' . str_replace('"', "'", (string) $oData['toVar']) . '"} = ';
		} else {
			$str = 'echo ';
		}

		$data = $this->prepareData($oData, "'");

		if (Module::isFront()) {
			unset($data['items']);
		}

		if (SystemConfig::load('useLatteTemplateReader')) {
			$str .= '$this->global->templateText->render("' . ($data['type'] ?? 'text') . '", ($this->getParameters()[\'templateTextValues\'] ?: ($this->global->getTemplateTextValues)($this->global))["' . $oData['name'] . '"] ?? "", \'' . Json::encode($data) . '\')';
		} else {
			$str .= '$this->global->templateText->render("' . ($data['type'] ?? 'text') . '", $templateTextValues["' . $oData['name'] . '"] ?? "", \'' . Json::encode($data) . '\')';
		}

		return $writer->write($str);
	}

	public function macroTemplateTextR(MacroNode $node, PhpWriter $writer): string
	{
		$data = self::parseInputAttrs($node->args);
		$data = $this->prepareData($data);

		$str = 'echo \'{tt';
		foreach ($data as $k => $v) {
			$str .= ' ' . $k . '="' . $v . '"';
		}

		$str .= '}\'';

		return $writer->write($str);
	}

	private function prepareData(array $data, string $quotes = "'"): array
	{
		$re = '/{\$([^}]+)}/';
		foreach ($data as $k => $v) {
			preg_match_all($re, (string) $v, $matches, PREG_OFFSET_CAPTURE, 0);

			foreach ($matches as $mk => $match) {
				$data[$k] = str_replace(
					(string) $matches[0][$mk][0],
					$quotes . " . $" . $matches[1][$mk][0] . " . " . $quotes,
					(string) $data[$k],
				);
			}

			if ($v && str_starts_with((string) $v, '{') && str_ends_with((string) $v, '}') && Arrays::isJson($v)) {
				$data[$k] = Json::decode($v, Json::FORCE_ARRAY);
			} else if ($k === 'items') {
				$data[$k] = str_replace("'", "\'", (string) $v);
			}
		}

		return $data;
	}

	public static function parseInputAttrs(string $str): array
	{
		$re  = '/(\w+)="(.*)"(?:\s|}$|\Z)/U';
		$arr = [];
		preg_match_all($re, $str, $matches, PREG_SET_ORDER, 0);
		foreach ($matches as $match) {
			/** @phpstan-ignore-next-line  */
			if (isset($match[1], $matches[2])) {
				if ($match[1] === 'multiLang' && $match[2] !== 'true') {
					continue;
				}

				$arr[$match[1]] = $match[2];
			}
		}

		return $arr;
	}
}
