<?php declare(strict_types = 1);

namespace Core\Model\Translation\Loaders;

use Contributte\Translation\Loaders\Neon;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Arrays;
use Core\Model\SystemConfig;
use Core\Model\Translation\Utils\Helpers;
use Doctrine\DBAL\DBALException;
use Nette\DI\Container;
use Nette\Utils\ArrayHash;
use stdClass;
use Symfony\Component\Translation\MessageCatalogue;

class NeonDoctrineMixedLoader extends Neon
{
	/** @var EntityManagerDecorator */
	protected $em;

	/** @var Container */
	protected $container;

	/** @var callable */
	protected $mergeCallable;

	/**
	 * NeonDoctrineMixedLoader constructor.
	 */
	public function __construct(EntityManagerDecorator $em, Container $container)
	{
		$this->em        = $em;
		$this->container = $container;
	}

	public function getDefaults(): array
	{
		$config = $this->getConfig('defaults');

		return [
			'table'   => $config['table'],
			'id'      => $config['id'],
			'locale'  => $config['locale'],
			'message' => $config['message'],
		];
	}

	/**
	 * @param mixed|null $default
	 */
	public function getConfig(string $key, $default = null): array
	{
		SystemConfig::setParams($this->container->getParameters()['system']['staticTextsTranslation']);

		return SystemConfig::load($key, $default);
	}

	/**
	 * @param mixed  $resource
	 * @param string $locale
	 * @param string $domain
	 */
	public function load($resource, $locale, $domain = 'messages'): MessageCatalogue
	{
		if (Helpers::isDomainAllowed($domain, $this->getConfig('domains'))) {
			$messages = $this->getMessages(
				ArrayHash::from($this->getDefaults()),
				$resource,
				$locale,
				$domain,
			);

			if (count($messages) > 0) {
				$fileResource = @file_get_contents($resource);
				if ($fileResource) {
					$arr       = Arrays::flattenWithDottedKeys(\Nette\Neon\Neon::decode($fileResource));
					$catalogue = new MessageCatalogue($locale);
					$catalogue->add(array_merge($arr, $messages), $domain);

					return $catalogue;
				}
			}
		}

		return parent::load($resource, $locale, $domain);
	}

	protected function getMessages(stdClass $config, string $resource, string $locale, string $domain): array
	{
		$messages = [];

		if (count($this->em->getConnection()->fetchAllAssociative('SHOW TABLES LIKE \'core__static_text_translation\'')) === 0) {
			return [];
		}

		$data = $this->container->getService('core.staticTexts')->merge();

		foreach ($data[$domain][$locale] ?? [] as $id => $message) {
			$messages[(string) str_replace($domain . '.', '', (string) $id)] = $message;
		}

		return $messages;
	}

}
