<?php declare(strict_types = 1);

namespace Core\Model\UI\Form\Controls;

use Core\Model\Helpers\Strings;
use Core\Model\Lang\Lang;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Traits\ChoiceInputTrait;
use Core\Model\UI\Form\Traits\TInputExtend;
use Nette\Forms\Controls\ChoiceControl;
use Nette\Utils\Html;

/**
 * @method BaseForm getForm(bool $throw = true)
 */
class BoolInput extends ChoiceControl
{
	use ChoiceInputTrait;
	use TInputExtend;

	protected array  $items            = [];
	protected string $textTrue         = 'default.yes';
	protected string $textFalse        = 'default.no';
	protected Html   $container;
	public bool      $disableJsWatcher = false;

	/**
	 * @param string|object $label
	 */
	public function __construct($label, array $items)
	{
		parent::__construct($label, []);
		$this->control->type = 'radio';
		$this->container     = Html::el('div');
		$this->setOption('type', 'radio');

		$this->setItems(empty($items) ? $this->getDefaultItems() : $items);
	}

	public function setTextTrue(string $text): self
	{
		$this->textTrue = $text;

		return $this;
	}

	public function setTextFalse(string $text): self
	{
		$this->textFalse = $text;

		return $this;
	}

	public function getDefaultItems(): array
	{
		$this->items = [
			1 => $this->translate($this->textTrue),
			0 => $this->translate($this->textFalse),
		];

		return $this->items;
	}

	public function getItems(): array
	{
		$arr = $this->rawItems;

		array_walk($arr, function(&$a) {
			if (is_array($a))
				$a = $this->translate($a['title']);
			else
				$a = $this->translate($a);
		});

		return $arr;
	}

	protected function isValueSelected(string|int|null $value, ?string $lang = null): bool
	{
		if (is_null($value)) {
			return false;
		}

		$val = $this->getValue();

		if ($lang) {
			if (is_array($val) && isset($val[$lang])) {
				return $val[$lang] == $value;
			} else {
				return $value == $val;
			}
		} else {
			return $value == $val;
		}
	}

	public function hasDefaultItems(): bool
	{
		$defaultItems = $this->getDefaultItems();
		$items        = $this->getItems();

		return $defaultItems[0] == $items[0] && $defaultItems[1] == $items[1] && count($items) == 2;
	}

	public function getValue()
	{
		if ($this->getIsMultiLanguage())
			return $this->value;
		else
			return array_key_exists($this->value, $this->getItems()) ? $this->value : ($this->hasDefaultItems(
			) ? 0 : null);
	}

	public function getControl()
	{
		$baseControl = parent::getControl();

		if (!$baseControl instanceof Html) {
			$baseControl = Html::fromHtml($baseControl);
		}

		$langs       = $this->getForm()->langsService->getLangs(false);
		$isMultilang = $this->getIsMultiLanguage();

		$items     = $this->getItems();
		$container = $this->container;

		$hasDefaultItems = $this->hasDefaultItems();

		$baseClass = 'switch' . ($hasDefaultItems ? ' switch--bool' : '');

		$defaultLang  = array_key_first($langs);
		$selectedLang = $_COOKIE['contentLang'] ?? $defaultLang;
		if (!array_key_exists($selectedLang, $langs)) {
			$selectedLang = $defaultLang;
		}

		foreach ($isMultilang ? $langs : ['' => null] as $k => $lang) {
			$wrap  = Html::el('div', [
				'class' => $baseClass,
			]);
			$slide = Html::el('div class=switch__slide');

			if ($hasDefaultItems) {
				$input        = clone $baseControl;
				$input->attrs = [...$input->attrs, 'class'    => 'switch__input', 'type' => 'checkbox', 'value' => 1,
				                                   'name'     => $this->getHtmlName() . ($lang ? "[$k]" : ''),
				                                   'id'       => $this->getHtmlId() . ($lang ? "-$k" : ''),
				                                   'checked'  => $this->isValueSelected(1, $k) ? true : false,
				                                   'disabled' => $this->isDisabled()];

				if ($this->disableJsWatcher)
					$input->setAttribute('data-disable-watcher', 'true');

				$label = Html::el('label', [
					'class' => 'switch__item',
					'for'   => $input->attrs['id'],
				])
					->addHtml(
						Html::el('span class=switch__item-1')
							->addHtml(Html::el('i class="fas fa-check"')),
					)
					->addHtml(Html::el('span class=switch__item-o'))
					->addHtml(
						Html::el('span class=switch__item-0')
							->addHtml(Html::el('i class="fas fa-times"')),
					);

				if ($lang) {
					$wrap->addHtml($input)
						->addHtml($label);
				} else {
					$container->addHtml($input)
						->addHtml($label);
				}
			} else {
				foreach ($items as $value => $caption) {
					$disabledOption = $this->isValueDisabled($value);
					$isSelected     = $this->isValueSelected($value, $k);

					$rawItem = $this->rawItems[$value];

					$input = Html::el('input', [
						'class'    => 'switch__input',
						'type'     => 'radio',
						'value'    => $value,
						'name'     => $this->getHtmlName() . ($lang ? "[$k]" : ''),
						'id'       => $this->getHtmlId() . ($lang ? "-$k" : '') . '-' . Strings::webalize(
								(string) $value,
							),
						'checked'  => $isSelected,
						'disabled' => $disabledOption,
					]);

					$label = Html::el('label', [
						'class'   => 'switch__item',
						'for'     => $input->attrs['id'],
						'data-bg' => is_array($rawItem) ? $rawItem['btnClass'] : ($value == 1 ? 'success' : 'danger'),
					])->setText($caption);

					if ($lang) {
						$wrap->addHtml($input)
							->addHtml($label);
					} else {
						$container->addHtml($input)
							->addHtml($label);
					}
				}

				if ($lang) {
					$style = '';
					if ($k !== $selectedLang) {
						$style = 'display: none;';
					}

					$wrap->addHtml($slide);
					$wrap = Html::el('div', [
						'data-content-lang' => $k,
						'style'             => $style,
					])->addHtml($wrap);
				} else {
					$container->addHtml($slide);
				}
			}

			if ($lang) {
				$style = '';
				if ($k !== $selectedLang) {
					$style = 'display: none;';
				}

				$wrap = Html::el('div', [
					'data-content-lang' => $k,
					'style'             => $style,
				])->addHtml($wrap);
				$container->addHtml($wrap);
			} else {
				$container->addClass($baseClass);
				$container = Html::el('div')->addHtml($container);
			}
		}

		return $container;
	}
}
