<?php declare(strict_types = 1);

namespace Core\Model\UI\Form\Controls;

use Core\Components\FilesManager\FilesManager as FilesManagerComponent;
use Core\Model\FilesManager\FilesManager;
use Core\Model\Lang\Langs;
use Core\Model\UI\AbstractPresenter;
use Core\Model\UI\ControllerHelper;
use Exception;
use Gallery\AdminModule\Components\Image\IImagesPickerFactory;
use Nette\Bridges\ApplicationLatte\TemplateFactory;
use Nette\DI\Container;
use Nette\Http\Request;
use Nette\Http\Session;
use Nette\Utils\Html;
use Nette\Utils\Json;
use stdClass;

class EditorInput extends TextInput
{
	protected static bool $assetsLoaded = false;

	/** @var callable */
	public $albumImagesCallback;

	protected ?IImagesPickerFactory $albumImagesPickerFactory = null;
	protected Session               $session;
	protected Request               $request;
	protected FilesManagerComponent $controller;

	protected ?string $controllerName = null;
	protected string  $toolbar        = 'Basic';
	protected array   $toolbarButtons = [];
	protected ?string $styles         = null;
	protected int     $height         = 500;
	protected bool    $disableAutoP   = false;
	protected array   $roots          = [];
	public array      $editorCfg      = [];

	public function __construct(
		string                     $label,
		protected Container        $container,
		protected FilesManager     $filesManager,
		protected ControllerHelper $controllerHelper,
		Session                    $session,
		Request                    $request,
		protected TemplateFactory  $templateFactory,
		protected Langs            $langs
	)
	{
		parent::__construct($label);
		$this->setHtmlType('hidden');
		$this->controller = new FilesManagerComponent($this->filesManager, $session, $request);
		$this->session    = $session;
		$this->request    = $request;

		$this->monitor(AbstractPresenter::class, function(AbstractPresenter $presenter) {
			$form = $this->getForm(false);

			if (!$form) {
				throw new Exception('Form missing');
			}
			$this->controllerName = 'filesManagerController' . ucfirst($this->getName());
			$currGroup            = $form->getCurrentGroup();

			$form->setCurrentGroup(null);
			if (!$form->getComponent($this->controllerName, false)) {
				$form->addComponent($this->controller, $this->controllerName);
			}
			if (!$form->getComponent('controllerHelper', false)) {
				$form->addComponent($this->controllerHelper, 'controllerHelper');
			}

			$form->setCurrentGroup($currGroup);
		});
	}

	public function handleElFinder(): void
	{
		$this->controller->render();
	}

	public function getControl(): Html
	{
		/** @var AbstractPresenter $presenter */
		$presenter = $this->lookup(AbstractPresenter::class, false);

		$this->controller->setModal();
		$wrapper   = Html::el('');
		$hidden    = parent::getControl();
		$maxLength = null;
		$langs     = $this->langs->getLangs(false);

		$baseValue = $this->getValue();
		if ($baseValue instanceof stdClass) {
			$baseValue = (array) $baseValue;
		}

		$editorIds = [];
		if ($this->getIsMultiLanguage()) {
			$defaultLang  = array_key_first($langs);
			$selectedLang = $_COOKIE['contentLang'] ?? $defaultLang;
			if (!array_key_exists($selectedLang, $langs)) {
				$selectedLang = $defaultLang;
			}

			$editor = Html::el();
			foreach ($hidden->getChildren() as $hChild) {
				$inputId     = $hChild->getAttribute('id');
				$lang        = $hChild->attrs['data-content-lang'];
				$ckEditorId  = str_replace('-', '', (string) $inputId) . 'Editor';
				$editorIds[] = ['input' => $inputId, 'editor' => $ckEditorId];
				$maxLength   = $hChild->getAttribute('maxlength');

				$style = '';
				if ($lang !== $selectedLang) {
					$style = 'display: none;';
				}

				$editor->addHtml(Html::el('textarea', [
					'id'                => $ckEditorId,
					'class'             => 'ck_textarea',
					'data-content-lang' => $lang,
					'style'             => $style,
				])->addHtml($baseValue[$lang] ?? ''));
			}
		} else {
			$inputId     = $hidden->getAttribute('id');
			$ckEditorId  = str_replace('-', '', (string) $inputId) . 'Editor';
			$editor      = Html::el('textarea', ['id' => $ckEditorId, 'class' => 'ck_textarea']);
			$editorIds[] = ['input' => $inputId, 'editor' => $ckEditorId];

			if ($baseValue) {
				$editor->addHtml($baseValue);
			}
		}

		$wrapper->addHtml($hidden);
		$wrapper->addHtml($editor);

		if (!$this->controller->getParent() && $presenter) {
			$presenter->addComponent($this->controller, $this->controllerName);
		}
		ob_start();
		$this->controller->render();
		$content = ob_get_clean();

		$latte = $this->templateFactory->createTemplate();
		$latte->setFile(__DIR__ . '/EditorInput.latte');

		$cfg = [
			'removeButtons' => 'Templates,Replace,Find,SelectAll,SpellChecker,CreateDiv,Anchor,HorizontalRule,SpecialChar,Iframe',
		];

		if ($this->disableAutoP) {
			$cfg                  += [
				'autoParagraph'         => false,
				'enterMode'             => 2, // CKEDITOR.ENTER_BR
				'shiftEnterMode'        => 2, // CKEDITOR.ENTER_BR
				'fillEmptyBlocks'       => false,
				'forcePasteAsPlainText' => true,
			];
			$cfg['removeButtons'] .= ',Format,Styles,Font,FontSize';
		} else {
			$cfg += [
				'autoParagraph' => true,
				'enterMode'     => 1, // CKEDITOR.ENTER_P
			];
		}

		if ($maxLength) {
			$cfg['wordcount'] = [
				'showCharCount'      => true,
				'countSpacesAsChars' => true,
				'countHTML'          => true,
				'maxCharCount'       => $maxLength,
			];
		}

		if ($this->albumImagesCallback) {
			$latte->preparedAlbumId = call_user_func($this->albumImagesCallback);
		}

		$wrapper->addHtml(
			$latte->renderToString(null, [
				'time'            => time(),
				'ckeditorScriptV' => BUILD_TIME,
				'editorIds'       => $editorIds,
				'controllerName'  => $this->controllerName,
				'elFinderUrl'     => $this->controller->link('showModal!', [$this->controllerName]),
				'elFinderOnlyUrl' => $this->controller->getPresenter()->link(':Core:Admin:FilesManager:only'),
				'toolbar'         => $this->toolbar,
				'toolbarButtons'  => $this->toolbarButtons,
				'styles'          => $this->styles,
				'height'          => $this->height,
				'width'           => 'auto',
				'disableAutoP'    => (bool) $this->disableAutoP,
				'roots'           => Json::encode($this->roots),
				'editorCfg'       => array_merge($cfg, $this->editorCfg),
			]),
		);

		$wrapper->addHtml((string) $content);

		return $wrapper;
	}

	public function setContainer(Container $container): void
	{
		$this->container = $container;
	}

	public function setToolbar(string $toolbar): self
	{
		$this->toolbar = $toolbar;

		return $this;
	}

	public function setToolbarButtons(string $buttons): self
	{
		$this->toolbarButtons = array_map(static fn($v) => trim($v), explode(',', $buttons));

		return $this;
	}

	public function setStyles(string $styles): self
	{
		$this->styles = $styles;

		return $this;
	}

	/**
	 * @param int|string $height
	 */
	public function setHeight($height): self
	{
		$this->height = (int) $height;

		return $this;
	}

	public function setDisableAutoP(bool $disable = true): self
	{
		$this->disableAutoP = $disable;

		return $this;
	}

	public function addRoot(string $path): self
	{
		$this->roots[] = $path;

		return $this;
	}

	public static function register(Container $systemContainer): void
	{
		$class = self::class;
		\Nette\Forms\Container::extensionMethod("addEditor", static function(
			\Nette\Forms\Container $container,
			                       $name,
			                       $caption,
		) use ($class, $systemContainer) {
			/** @var EditorInput $comp */
			$comp = new $class(
				$caption,
				$systemContainer,
				$systemContainer->getByType(FilesManager::class, false),
				$systemContainer->getByType(ControllerHelper::class, false),
				$systemContainer->getByType(Session::class, false),
				$systemContainer->getByType(Request::class, false),
				$systemContainer->getByType(TemplateFactory::class, false),
				$systemContainer->getByType(Langs::class, false)
			);

			$comp->albumImagesPickerFactory = $systemContainer->getByType(IImagesPickerFactory::class, false);

			$container->addComponent($comp, $name);

			return $comp;
		});
	}

	/*******************************************************************************************************************
	 * ==================  Album images
	 */

	public function handleAlbumImages(): void
	{
		/** @var AbstractPresenter $presenter */
		$presenter = $this->lookup(AbstractPresenter::class, false);
		if (!$presenter) {
			throw new Exception('Presenter missing');
		}

		$control = $this->albumImagesPickerFactory->create();
		$control->setParent($presenter);
		$control->setByAlbum($this->request->getPost('albumId'));
		ob_start();
		$control->render();
		$content = ob_get_clean();
		$presenter->sendJson($content);
	}

	public function setAlbumImagesCallback(callable $callback): self
	{
		$this->albumImagesCallback = $callback;

		return $this;
	}
}
