<?php declare(strict_types = 1);

namespace Core\Model\UI\Form\Controls;

use Contributte\Translation\Translator;
use Core\Model\SystemConfig;
use Core\Model\UI\Form\BaseForm;
use libphonenumber\NumberParseException;
use libphonenumber\PhoneNumber;
use libphonenumber\PhoneNumberUtil;
use Nette\Forms\IControl;
use Nette\Utils\Html;

class PhoneInput extends TextInput
{
	protected bool $enableValidation = true;
	final public const PHONE = 'Core\Model\UI\Form\Controls\PhoneInput::validatePhone';
	public static string $phoneMessages = 'default.formMessages.phone';

	/**
	 * @param string|object $label
	 */
	public function __construct($label = null)
	{
		parent::__construct($label, null);

		$this->control->type = 'tel';
		$this->setOption('type', 'tel');

		$this->setPlaceholder('+420 123 456 789');
	}

	public function validate(): void
	{
		if (!$this->enableValidation) {
			return;
		}

		$this->addCondition(BaseForm::FILLED)
			->addRule(BaseForm::PATTERN, 'default.formMessages.phone_min', '(.*\d.*){9}');

		parent::validate();
	}

	public static function validatePhone(IControl $control, ?array $regions = null): bool
	{
		$valid = false;

		if (!$regions) {
			$regions = SystemConfig::load('phoneInputRegions');
		}
		$phoneUtil = PhoneNumberUtil::getInstance();

		try {
			foreach ($regions as $region) {
				/** @var PhoneNumber $phoneParsed */
				$phoneParsed = $phoneUtil->parse($control->getValue(), $region);
				$valid       = $phoneUtil->isValidNumber($phoneParsed);

				if ($valid) {
					break;
				}
			}
		} catch (NumberParseException) {
			$valid = false;
		}

		return $valid;
	}

	public function getControl(): Html
	{
		/** @var null|Translator $translator */
		$translator = $this->getTranslator();
		if ($translator) {
			$locale = $translator->getLocale();

			switch ($locale) {
				case 'sk':
					$this->setPlaceholder('+421 123 456 789');
					break;
				case 'de':
					$this->setPlaceholder('+49 123 12345678');
					break;
				default:
					$this->setPlaceholder('+420 123 456 789');
					break;
			}
		}

		return parent::getControl();
	}

	public function enableValidation(bool $val = true): self
	{
		$this->enableValidation = $val;

		return $this;
	}
}
