<?php declare(strict_types = 1);

namespace Core\Model\UI;

use Core\Model\Event\EventDispatcher;
use Core\AdminModule\Components\ContentLangSwitcher\IContentLangSwitcherFactory;
use Core\Model\Event\ControlEvent;
use Core\Model\Helpers\CacheHelper;
use Core\Model\Images\ImagePipe;
use Core\Model\Lang\Langs;
use Core\Model\Settings;
use Core\Model\SystemConfig;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use Core\Model\Entities\EntityManagerDecorator;
use Nette\ComponentModel\IComponent;
use Nette\ComponentModel\IContainer;
use Nette\Localization\ITranslator;
use Nette\Application\UI\Control;
use Nette\Application\UI\Presenter;
use Nette\Caching\Cache;
use Nette\Caching\IStorage;
use Core\Model\Helpers\Strings;

abstract class BaseControl extends Control
{
	/** @var EventDispatcher @inject */
	public $eventDispatcher;

	/** @var IContentLangSwitcherFactory @inject */
	public $contentLangSwitcherFactory;

	/** @var SystemConfig @inject */
	public $configService;

	/** @var Settings @inject */
	public $settings;

	/** @var ITranslator */
	protected $translator;

	/** @var EntityManagerDecorator */
	protected $em;

	/** @var IStorage */
	protected $cacheStorage;

	/** @var ImagePipe */
	protected $imagePipe;

	/** @var Cache */
	protected $cache;

	/** @var Langs */
	protected $langsService;

	/** @var Cache */
	protected $templateFileCache;

	/** @var array */
	protected $handleCallbacks = [];

	/**
	 * @param bool $throw
	 *
	 * @return Presenter|AbstractPresenter|\Nette\ComponentModel\IComponent|null
	 */
	public function getPresenter($throw = true): ?Presenter
	{
		return $this->lookup(AbstractPresenter::class, $throw);
	}

	public function setParent(?IContainer $parent, string $name = null)
	{
		$parent = parent::setParent($parent, $name);

		$this->monitor(AbstractPresenter::class, function() use ($parent) {
			$this->attached($parent->getPresenter());
		});

		return $parent;
	}

	protected function attached($presenter): void
	{
		parent::attached($presenter);

		if ($this->eventDispatcher) {
			$class      = get_called_class();
			$isOverride = strpos($class, 'Override') !== false;
			$event      = new ControlEvent($this);

			if ($isOverride)
				$this->eventDispatcher->dispatch($event, str_replace('Override', '', $class) . '::onAttach');
			$this->eventDispatcher->dispatch($event, $class . '::onAttach');
		}
	}

	public function handleCallback(string $key, ?array $args = null)
	{
		$args = func_get_args();
		$c    = $this->handleCallbacks[$key] ?? null;

		if ($c && is_callable($c))
			if ($args[1] !== null)
				$c($args[1]);
			else
				$c();
	}

	/*******************************************************************************************************************
	 * =======================     Helpers
	 */
	public function t($message, $count = null, $parameters = [], $domain = null, $locale = null)
	{
		return $this->translator->translate($message, $count, $parameters, $domain, $locale);
	}

	/**
	 * Vraci naprimo POST pole od aktualniho presenteru a requestu
	 *
	 * @param null $key
	 * @param null $default
	 *
	 * @return mixed
	 */
	public function getPost($key = null, $default = null)
	{
		return $this->getPresenter()->getHttpRequest()->getPost($key, $default);
	}

	/*******************************************************************************************************************
	 * =======================     Setters
	 */

	public function setTranslator(ITranslator $translator)
	{
		$this->translator = $translator;
	}

	public function setEntityManager(EntityManagerDecorator $entityManager)
	{
		$this->em = $entityManager;
	}

	public function setCacheStorage(IStorage $cacheStorage)
	{
		$this->cacheStorage = $cacheStorage;
	}

	public function setImagePipe(ImagePipe $imagePipe)
	{
		$this->imagePipe  = $imagePipe;
		$this->onAnchor[] = function($control) {
			$control->template->_imagePipe = $this->imagePipe;
		};
	}

	public function setLangsService(Langs $langs)
	{
		$this->langsService = $langs;
	}

	public function addHandleCallback(string $key, $c): self
	{
		$this->handleCallbacks[$key] = $c;

		return $this;
	}

	/*******************************************************************************************************************
	 * =======================     Getters
	 */

	protected function getTemplateDirs()
	{
		$list = [];

		$path = explode(':', $this->getPresenter()->getName());
		if (count($path) > 2) {
			$tmp     = $path[1];
			$path[1] = $path[0];
			$path[0] = $tmp;
		}

		//TODO možnost měnit šablonu
		$module = array_shift($path);
		do {
			$pathString = implode(DIRECTORY_SEPARATOR, $path);

			$list[] = Strings::joinPaths(TEMPLATES_DIR, $module, $this->template->currentSite->getIdent(), $pathString, 'Components');
			$list[] = Strings::joinPaths(TEMPLATES_DIR, $module, 'default', $pathString, 'Components');
		} while (array_pop($path));

		$reflection = $this->getReflection();
		foreach ([$reflection, $reflection->getParentClass()] as $ref) {
			$reflectionDir   = dirname($ref->getFileName());
			$tmp             = explode(DIRECTORY_SEPARATOR, $reflectionDir);
			$moduleComponent = [];

			foreach ($tmp as $k => $v) {

				if ($v == 'src') {
					if ($tmp[$k + 1] == 'Model')
						break 2;

					$moduleComponent[] = $v;
					break;
				}

				$moduleComponent[] = $v;
			}

			$srcPath = implode(DIRECTORY_SEPARATOR, $moduleComponent);
			$list[]  = $srcPath . DIRECTORY_SEPARATOR . $module . 'Module' . str_replace($srcPath, '', $reflectionDir);
			$list[]  = $srcPath . DIRECTORY_SEPARATOR . $module . 'Module' . str_replace($srcPath, '', dirname($reflectionDir));
			$list[]  = $reflectionDir;
			$list[]  = dirname($reflectionDir);
		}

		$list = array_merge($list, $this->getTemplateDirsInSrc());

		return $list;
	}

	protected function getTemplateDirsInSrc()
	{
		$list          = [];
		$reflectionDir = dirname($this->getReflection()->getFileName());
		if (strpos($reflectionDir, DS . 'custom' . DS) !== false) {
			$reflectionDir = explode(DS . 'custom', $reflectionDir);
			$reflectionDir = SRC_DIR . $reflectionDir[1];
			$list[]        = $reflectionDir;
			$list[]        = dirname($reflectionDir);
		}

		return $list;
	}

	/**
	 * @param null $filename
	 * @param null $name
	 *
	 * @return string|null
	 * @throws \ReflectionException
	 */
	protected function getTemplateFile($filename = null, $name = null)
	{
		$reflect                        = new \ReflectionClass($this);
		$this->template->originTemplate = str_replace('.php', '.latte', $reflect->getFileName());

		$name = $name ?: $this->getName();
		if ($filename) {
			$filename = ucfirst($filename);
			$filename = ucfirst($name) . '/' . preg_replace('/Override$/', '', $filename);
		} else {
			$filename = ucfirst($name);
		}

		$key  = CacheHelper::getComponentKey($this->getPresenter(), $this, $filename);
		$file = CacheHelper::getTemplatesCache()->load($key, function(&$dep) use ($filename, $reflect) {
			$dep = [
				Cache::EXPIRE => '1 week',
			];

			foreach ([$filename, $reflect->getShortName()] as $name) {
				// Hledání šablony v základních složkách
				foreach ($this->getTemplateDirs() as $dir) {
					$file = $dir . '/' . $name . '.latte';

					if (file_exists($file))
						return $file;
				}

				// Pokud se nenajde, zkontrolovat zanořené v src
				foreach ($this->getTemplateDirsInSrc() as $dir) {
					foreach (glob($dir . '/**/' . $name . '.latte') as $f)
						return $f;
				}
			}

			return null;
		});

		return $file ?: null;
	}

	/*******************************************************************************************************************
	 * =======================     Controls
	 */

	/**
	 * Upřednostňovaný a obecný způsob vytváření Nette Formu
	 * s překladačem
	 * @return BaseForm
	 */
	protected function createForm()
	{
		$form = new BaseForm();
		$form->setTranslator($this->translator);
		$form->setLangsService($this->langsService);

		return $form;
	}

	/**
	 * Upřednostňovaný a obecný způsob vytváření DataGrid
	 * s překladačem
	 * @return BaseDataGrid
	 */
	protected function createGrid()
	{
		$grid = new BaseDataGrid();
		$grid->setTranslator($this->translator);
		$grid->setStrictSessionFilterValues(false);

		return $grid;
	}

	/*******************************************************************************************************************
	 * ========================  Components
	 */

	protected function createComponentContentLangSwitcher()
	{
		return $this->contentLangSwitcherFactory->create();
	}
}
